package cn.breeze.elleai.application.service;

import cn.breeze.elleai.application.dto.PageResult;
import cn.breeze.elleai.application.dto.request.*;
import cn.breeze.elleai.application.dto.response.*;
import cn.breeze.elleai.domain.sparring.model.request.QaAssistantRequestModel;
import cn.breeze.elleai.domain.sparring.model.request.QaAssistantSaveModel;
import cn.breeze.elleai.domain.sparring.model.request.UserChatCompletionSaveModel;
import cn.breeze.elleai.domain.sparring.model.request.UserQaRequestModel;
import cn.breeze.elleai.domain.sparring.model.response.QaAssistantResponseModel;
import cn.breeze.elleai.domain.sparring.model.response.UserChatCompletionHistoryResponseModel;
import cn.breeze.elleai.domain.sparring.model.response.UserChatCompletionResponseModel;
import cn.breeze.elleai.domain.sparring.service.ChatCompletionService;
import cn.breeze.elleai.util.UserPrincipal;
import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.UUID;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSONObject;
import com.mybatisflex.core.paginate.Page;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.util.*;
import java.util.stream.Collectors;


/**
 * 应用服务
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class AppChatCompletionService {

    @Value("${dify.api_base:https://ai-api.tech.breezeai.cn/v1}")
    private String difyBase;

    @Value("${dify.api_key:app-ilgoEphXjLw0I7x7fGeCyKYk}")
    private String apiKey;

    private final RestTemplate restTemplate = new RestTemplate();

    private final ChatCompletionService chatCompletionService;


    /************************************************** AI助手 **************************************************/

    /**
     * 更新助手
     * @param id
     * @param status
     */
    public void updateQaAssistantStatus(Integer id, Integer status) {
        chatCompletionService.updateQaAssistantStatus(id, status);
    }

    /**
     * 删除助手
     * @param id
     */
    public void deleteQaAssistant(Integer id) {
        chatCompletionService.deleteQaAssistant(id);
    }

    /**
     * 获取助手详情
     * @param id
     * @return
     */
    public QaAssistantDto qaAssistantDetail(Integer id) {
        QaAssistantResponseModel model = chatCompletionService.qaAssistantDetail(id);
        return BeanUtil.copyProperties(model, QaAssistantDto.class);
    }


    /**
     * 保存助手
     * @param dto
     */
    public void saveQaAssistant(QaAssistantSaveDto dto) {

        QaAssistantSaveModel model = BeanUtil.copyProperties(dto, QaAssistantSaveModel.class);
        model.setCategoryIds(JSONObject.toJSONString(dto.getCategoryIdList()));
        model.setVisibleRoleIds(JSONObject.toJSONString(dto.getVisibleRoleList()));
        chatCompletionService.saveQaAssistant(model);
    }


    /**
     * 助手分页查询
     * @param request
     * @return
     */
    public PageResult<QaAssistantDto> qaAssistantPaginQuery(QaAssistantRequestDto request) {

        QaAssistantRequestModel requestModel = BeanUtil.copyProperties(request, QaAssistantRequestModel.class);

        Page<QaAssistantResponseModel> page = chatCompletionService.qaAssistantPaginQuery(requestModel);

        PageResult<QaAssistantDto> pageResult = PageResult.of(request.getPageNo(), request.getPageSize(), (int) page.getTotalRow(), null);

        if(CollectionUtil.isNotEmpty(page.getRecords())) {
            List<QaAssistantDto> dtoList = page.getRecords().stream().map(v -> {
                QaAssistantDto dto = BeanUtil.copyProperties(v, QaAssistantDto.class);

                if(StrUtil.isNotEmpty(v.getCategoryIds())) {
                    dto.setCategoryIdList(JSONObject.parseArray(v.getCategoryIds(), String.class));
                }
                if(StrUtil.isNotEmpty(v.getVisibleRoleIds())) {
                    dto.setVisibleRoleList(JSONObject.parseArray(v.getVisibleRoleIds(), String.class));
                }
                return dto;
            }).collect(Collectors.toList());

            pageResult.setItems(dtoList);
        }
        return pageResult;
    }


    /**
     * 助手列表查询
     * @param request
     * @return
     */
    public List<QaAssistantDto> qaAssistantList(QaAssistantRequestDto request) {

        List<QaAssistantDto> resultList = new ArrayList<>();

        QaAssistantRequestModel requestModel = BeanUtil.copyProperties(request, QaAssistantRequestModel.class);

        List<QaAssistantResponseModel> modelList = chatCompletionService.qaAssistantList(requestModel);

        if(CollectionUtil.isNotEmpty(modelList)) {
            resultList = modelList.stream().map(v -> {
                QaAssistantDto dto = BeanUtil.copyProperties(v, QaAssistantDto.class);

                if(StrUtil.isNotEmpty(v.getCategoryIds())) {
                    dto.setCategoryIdList(JSONObject.parseArray(v.getCategoryIds(), String.class));
                }
                if(StrUtil.isNotEmpty(v.getVisibleRoleIds())) {
                    dto.setVisibleRoleList(JSONObject.parseArray(v.getVisibleRoleIds(), String.class));
                }
                return dto;
            }).collect(Collectors.toList());
        }
        return resultList;
    }


    /************************************************** AI问答会话 **************************************************/

    /**
     * 获取AI问答会话详情
     * @param sessionId
     * @return
     */
    public List<UserChatCompletionHistoryDto> userQaDetail(Integer sessionId) {
        List<UserChatCompletionHistoryDto> resultList = new ArrayList<>();

        List<UserChatCompletionHistoryResponseModel> models = chatCompletionService.userQaHistoryDetail(sessionId);
        if(CollectionUtil.isNotEmpty(models)) {
            resultList = models.stream().map(v -> {
                UserChatCompletionHistoryDto dto = BeanUtil.copyProperties(v, UserChatCompletionHistoryDto.class);
                return dto;
            }).collect(Collectors.toList());
        }

        return resultList;
    }


    /**
     * AI问答会话分页查询
     * @param request
     * @return
     */
    public PageResult<UserChatCompletionDto> userQaPaginQuery(UserQaRequestDto request) {

        UserQaRequestModel requestModel = BeanUtil.copyProperties(request, UserQaRequestModel.class);

        Page<UserChatCompletionResponseModel> page = chatCompletionService.userQaPaginQuery(requestModel);

        PageResult<UserChatCompletionDto> pageResult = PageResult.of(request.getPageNo(), request.getPageSize(), (int) page.getTotalRow(), null);

        if(CollectionUtil.isNotEmpty(page.getRecords())) {
            List<UserChatCompletionDto> dtoList = page.getRecords().stream().map(v -> {
                UserChatCompletionDto dto = BeanUtil.copyProperties(v, UserChatCompletionDto.class);

                //TODO 员工名称，提问信息

                return dto;
            }).collect(Collectors.toList());

            pageResult.setItems(dtoList);
        }
        return pageResult;
    }


    /**
     * AI问答会话列表查询
     * @param request
     * @return
     */
    public PageResult<UserChatCompletionDto> userQaMobilePaginQuery(UserQaRequestDto request) {

        UserQaRequestModel requestModel = BeanUtil.copyProperties(request, UserQaRequestModel.class);

        Page<UserChatCompletionResponseModel> page = chatCompletionService.userQaPaginQuery(requestModel);

        PageResult<UserChatCompletionDto> pageResult = PageResult.of(request.getPageNo(), request.getPageSize(), (int) page.getTotalRow(), null);

        if(CollectionUtil.isNotEmpty(page.getRecords())) {
            List<UserChatCompletionDto> dtoList = page.getRecords().stream().map(v -> {
                UserChatCompletionDto dto = BeanUtil.copyProperties(v, UserChatCompletionDto.class);
                return dto;
            }).collect(Collectors.toList());

            pageResult.setItems(dtoList);
        }
        return pageResult;
    }

    /**
     * AI问答会话分页查询
     * @param request
     * @return
     */
    public PageResult<UserChatCompletionMobileDto> userQaHistoryPaginQuery(UserPrincipal userPrincipal, UserQaHistoryRequestDto request) {

        UserQaRequestDto req = new UserQaRequestDto();
        req.setUserId(userPrincipal.getUserId());

        PageResult<UserChatCompletionDto> userChatCompletionPageResult = userQaMobilePaginQuery(req);

        PageResult<UserChatCompletionMobileDto> pageResult = PageResult.of(userChatCompletionPageResult.getPageNo(), userChatCompletionPageResult.getPageSize(), userChatCompletionPageResult.getTotal(), null);

        if(Objects.nonNull(userChatCompletionPageResult) && CollectionUtil.isNotEmpty(userChatCompletionPageResult.getItems())) {

            List<UserChatCompletionMobileDto> items = userChatCompletionPageResult.getItems().stream().map(v -> {
                UserChatCompletionMobileDto dto = BeanUtil.copyProperties(v, UserChatCompletionMobileDto.class);
                dto.setContent(v.getFirstQuestion());
                return dto;
            }).collect(Collectors.toList());

            BeanUtil.copyProperties(userChatCompletionPageResult, pageResult);
            pageResult.setItems(items);
        }

        return pageResult;
    }

    /**
     * 移动端历史会话明细
     * @param userPrincipal
     * @param chatCompletionId
     * @param pageNo
     * @param pageSize
     * @return
     */
    public PageResult<UserChatCompletionHistoryMobileDto> userQaHistoryMobilePaginQuery(UserPrincipal userPrincipal, Integer chatCompletionId, Integer pageNo, Integer pageSize) {

        pageNo = Objects.isNull(pageNo) ? 1 : pageNo;
        pageSize = Objects.isNull(pageSize) ? 10 : pageSize;

        UserQaRequestDto req = new UserQaRequestDto();
        req.setUserId(userPrincipal.getUserId());

        Page<UserChatCompletionHistoryResponseModel> historyDetailPageResult = chatCompletionService.userQaHistoryDetailPaginQuery(chatCompletionId, pageNo, pageSize);

        PageResult<UserChatCompletionHistoryMobileDto> pageResult = PageResult.of(pageNo, pageSize, (int) historyDetailPageResult.getTotalRow(), null);

        if(Objects.nonNull(historyDetailPageResult) && CollectionUtil.isNotEmpty(historyDetailPageResult.getRecords())) {

            List<UserChatCompletionHistoryMobileDto> items = historyDetailPageResult.getRecords().stream().map(v -> {
                UserChatCompletionHistoryMobileDto dto = BeanUtil.copyProperties(v, UserChatCompletionHistoryMobileDto.class);
                return dto;
            }).collect(Collectors.toList());
            pageResult.setItems(items);
        }

        return pageResult;
    }

    /**
     * 用户提问
     * @param request
     * @return
     */
    public UserAskResultMobileDto userAsk(UserPrincipal userPrincipal, UserQaMobileRequestDto request) {
        UserAskResultMobileDto result = new UserAskResultMobileDto();

        // 更新会话信息
        UserChatCompletionSaveModel saveModel = new UserChatCompletionSaveModel();
        saveModel.setUserId(userPrincipal.getUserId());
        saveModel.setUserName(userPrincipal.getUserName());
        saveModel.setId(request.getChatCompletionId());
        saveModel.setLastQuestion(request.getContent());
        saveModel.setShopId(userPrincipal.getShopId());
        saveModel.setShopName(userPrincipal.getShopName());
        // 首次提问
        if(Objects.isNull(request.getChatCompletionId())) {
            saveModel.setCreateTime(new Date());
            saveModel.setFirstQuestion(request.getContent());
            saveModel.setSessionId(UUID.randomUUID().toString());
        }
        Integer recordId = chatCompletionService.saveUserQaSession(saveModel);

        // 保存问答详情
        chatCompletionService.saveUserQaRecord(recordId, 0, request.getContent());

        // 问一下AI
        String replyContent = ask4Knowledge(saveModel.getSessionId(), saveModel.getUserId(), request.getContent());

        // 保存AI问答详情
        chatCompletionService.saveUserQaRecord(recordId, 1, replyContent);

        result.setChatCompletionId(recordId);
        result.setReplyContent(replyContent);
        return result;
    }


    public String ask4Knowledge(String sessionId, String userId, String question) {
        if(true) {
            return "moke ai reply.";
        }

        Map<String, String> inputs = new HashMap<>();
        inputs.put("question", question);
        JSONObject param = new JSONObject();
        param.put("inputs", inputs);
        param.put("response_mode", "blocking");
        param.put("conversation_id", "");
        param.put("user", userId);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.setBearerAuth(apiKey);

        log.info("异步请求参数1，sessionId = {}", sessionId);
        log.info("异步请求参数2，req = {}", JSONObject.toJSONString(param));
        HttpEntity<String> postEntity = new HttpEntity<>(param.toJSONString(), headers);
        ResponseEntity<String> response = restTemplate.postForEntity(difyBase + "/chat-messages", postEntity, String.class);
        String body = response.getBody();
        log.info("执行结果：", JSONObject.parseObject(body));
        return body;
    }
}
