package cn.breeze.elleai.application.service;

import cn.breeze.elleai.application.dto.PageResult;
import cn.breeze.elleai.application.dto.inner.AiSingleEvaluateResultDto;
import cn.breeze.elleai.application.dto.inner.ExamineBusinessCacheDto;
import cn.breeze.elleai.application.dto.request.*;
import cn.breeze.elleai.application.dto.response.*;
import cn.breeze.elleai.domain.sparring.model.request.*;
import cn.breeze.elleai.domain.sparring.model.response.*;
import cn.breeze.elleai.domain.sparring.service.CommonService;
import cn.breeze.elleai.domain.sparring.service.ExamineService;
import cn.breeze.elleai.exception.InternalException;
import cn.breeze.elleai.util.Codes;
import cn.breeze.elleai.util.UserPrincipal;
import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.lang.UUID;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSONObject;
import com.mybatisflex.core.paginate.Page;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import static java.util.stream.Collectors.*;


/**
 * 应用服务
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class AppExamineService {

    private static String EXAM_BUSINESS_REDIS_KEY = "examine:business:%s"; // 考试流水缓存信息(businessNo)

    private static String EXAM_AI_TASK_REDIS_KEY = "examine:ai_task:%s"; // 考试AI点评任务执行状态(businessNo)



    private final ExamineService examineService;

    private final StringRedisTemplate redisTemplate;

    private final AiPlatformExtensionService extensionService;

    private final CommonService commonService;


    /************************************************** 场景分类 **************************************************/

    /**
     * 更新场景分类状态
     * @param id
     * @param status
     */
    public void updateSceneCategoryStatus(Integer id, Integer status) {
        examineService.updateSceneCategoryStatus(id, status);
    }

    /**
     * 删除场景分类
     * @param id
     */
    public void deleteSceneCategory(Integer id) {
        examineService.deleteSceneCategory(id);
    }

    /**
     * 获取场景分类详情
     * @param id
     * @return
     */
    public ExamineSceneCategoryDto sceneCategoryDetail(Integer id) {
        ExamineSceneCategoryResponseModel model = examineService.sceneCategoryDetail(id);
        return BeanUtil.copyProperties(model, ExamineSceneCategoryDto.class);
    }


    /**
     * 保存场景分类
     * @param dto
     */
    public void saveExamineSceneCategory(ExamineSceneCategorySaveDto dto) {

        Assert.notNull(dto.getSceneId(), "场景ID不能为空");

        ExamineSceneResponseModel scene = examineService.sceneDetail(dto.getSceneId());
        Assert.notNull(scene, "场景信息获取异常");

        ExamineSceneCategorySaveModel model = BeanUtil.copyProperties(dto, ExamineSceneCategorySaveModel.class);
        model.setVisibleRoleIds(JSONObject.toJSONString(dto.getVisibleRoleList()));
        examineService.saveSceneCategory(model);
    }


    /**
     * 场景分类分页查询
     * @param request
     * @return
     */
    public PageResult<ExamineSceneCategoryDto> examineSceneCategoryPaginQuery(ExamineSceneCategoryRequestDto request) {

        ExamineSceneCategoryRequestModel requestModel = BeanUtil.copyProperties(request, ExamineSceneCategoryRequestModel.class);

        Page<ExamineSceneCategoryResponseModel> page = examineService.sceneCategoryPaginQuery(requestModel);

        PageResult<ExamineSceneCategoryDto> pageResult = PageResult.of(request.getPageNo(), request.getPageSize(), (int) page.getTotalRow(), null);

        if(CollectionUtil.isNotEmpty(page.getRecords())) {
            List<ExamineSceneCategoryDto> dtoList = page.getRecords().stream().map(v -> {
                ExamineSceneCategoryDto dto = BeanUtil.copyProperties(v, ExamineSceneCategoryDto.class);

                if(Objects.nonNull(dto.getSceneId())) {
                    ExamineSceneResponseModel sceneResponseModel = examineService.sceneDetail(dto.getSceneId());
                    if(Objects.nonNull(sceneResponseModel)) {
                        dto.setSceneName(sceneResponseModel.getName());
                    }
                }

                if(StrUtil.isNotEmpty(v.getVisibleRoleIds())) {
                    dto.setVisibleRoleList(JSONObject.parseArray(v.getVisibleRoleIds(), String.class));
                }

                return dto;
            }).collect(Collectors.toList());

            pageResult.setItems(dtoList);
        }
        return pageResult;
    }

    /**
     * 场景分类分页查询
     * @param request
     * @return
     */
    public PageResult<ExamineSceneCategoryMobileDto> examineSceneCategoryMobilePaginQuery(ExamineSceneCategoryRequestDto request) {

        ExamineSceneCategoryRequestModel requestModel = BeanUtil.copyProperties(request, ExamineSceneCategoryRequestModel.class);
        requestModel.setStatus(1);

        Page<ExamineSceneCategoryResponseModel> page = examineService.sceneCategoryPaginQuery(requestModel);

        PageResult<ExamineSceneCategoryMobileDto> pageResult = PageResult.of(request.getPageNo(), request.getPageSize(), (int) page.getTotalRow(), null);

        if(CollectionUtil.isNotEmpty(page.getRecords())) {
            List<ExamineSceneCategoryMobileDto> dtoList = page.getRecords().stream().map(v -> {
                ExamineSceneCategoryMobileDto dto = BeanUtil.copyProperties(v, ExamineSceneCategoryMobileDto.class);
                return dto;
            }).collect(Collectors.toList());

            pageResult.setItems(dtoList);
        }
        return pageResult;
    }

    /**
     * 场景分类列表查询
     * @param request
     * @return
     */
    public List<ExamineSceneCategoryDto> examineSceneCategoryList(ExamineSceneCategoryRequestModel request) {

        List<ExamineSceneCategoryDto> resultList = new ArrayList<>();

        List<ExamineSceneCategoryResponseModel> modelList = examineService.sceneCategoryList(request);

        if(CollectionUtil.isNotEmpty(modelList)) {
            resultList = modelList.stream().map(v -> {
                ExamineSceneCategoryDto dto = BeanUtil.copyProperties(v, ExamineSceneCategoryDto.class);

                if(Objects.nonNull(dto.getSceneId())) {
                    ExamineSceneResponseModel sceneResponseModel = examineService.sceneDetail(dto.getSceneId());
                    if(Objects.nonNull(sceneResponseModel)) {
                        dto.setSceneName(sceneResponseModel.getName());
                    }
                }

                if(StrUtil.isNotEmpty(v.getVisibleRoleIds())) {
                    dto.setVisibleRoleList(JSONObject.parseArray(v.getVisibleRoleIds(), String.class));
                }

                return dto;
            }).collect(Collectors.toList());
        }
        return resultList;
    }

    /************************************************** 场景 **************************************************/

    /**
     * 更新场景
     * @param id
     * @param status
     */
    public void updateSceneStatus(Integer id, Integer status) {
        examineService.updateSceneStatus(id, status);
    }

    /**
     * 删除场景
     * @param id
     */
    public void deleteScene(Integer id) {
        examineService.deleteScene(id);
    }

    /**
     * 获取场景详情
     * @param id
     * @return
     */
    public ExamineSceneDto sceneDetail(Integer id) {
        ExamineSceneResponseModel model = examineService.sceneDetail(id);
        return BeanUtil.copyProperties(model, ExamineSceneDto.class);
    }


    /**
     * 保存场景
     * @param dto
     */
    public void saveExamineScene(ExamineSceneSaveDto dto) {

        ExamineSceneSaveModel model = BeanUtil.copyProperties(dto, ExamineSceneSaveModel.class);
        examineService.saveScene(model);
    }


    /**
     * 场景分类分页查询
     * @param request
     * @return
     */
    public PageResult<ExamineSceneDto> examineScenePaginQuery(ExamineSceneRequestDto request) {

        ExamineSceneRequestModel requestModel = BeanUtil.copyProperties(request, ExamineSceneRequestModel.class);

        Page<ExamineSceneResponseModel> page = examineService.scenePaginQuery(requestModel);

        PageResult<ExamineSceneDto> pageResult = PageResult.of(request.getPageNo(), request.getPageSize(), (int) page.getTotalRow(), null);

        if(CollectionUtil.isNotEmpty(page.getRecords())) {
            List<ExamineSceneDto> dtoList = page.getRecords().stream().map(v -> {
                ExamineSceneDto dto = BeanUtil.copyProperties(v, ExamineSceneDto.class);
                return dto;
            }).collect(Collectors.toList());

            pageResult.setItems(dtoList);
        }
        return pageResult;
    }


    /**
     * 场景分类分页查询
     * @param request
     * @return
     */
    public PageResult<ExamineSceneMobileDto> examineMobileScenePaginQuery(ExamineSceneMobileRequestDto request) {

        ExamineSceneRequestModel requestModel = BeanUtil.copyProperties(request, ExamineSceneRequestModel.class);
        // 移动端默认查询启用状态
        requestModel.setStatus(1);

        Page<ExamineSceneResponseModel> page = examineService.scenePaginQuery(requestModel);

        PageResult<ExamineSceneMobileDto> pageResult = PageResult.of(request.getPageNo(), request.getPageSize(), (int) page.getTotalRow(), null);

        if(CollectionUtil.isNotEmpty(page.getRecords())) {
            List<ExamineSceneMobileDto> dtoList = page.getRecords().stream().map(v -> {
                ExamineSceneMobileDto dto = BeanUtil.copyProperties(v, ExamineSceneMobileDto.class);
                return dto;
            }).collect(Collectors.toList());

            pageResult.setItems(dtoList);
        }
        return pageResult;
    }

    /**
     * 场景分类列表查询
     * @param request
     * @return
     */
    public List<ExamineSceneDto> examineSceneList(ExamineSceneRequestModel request) {

        List<ExamineSceneDto> resultList = new ArrayList<>();

        List<ExamineSceneResponseModel> modelList = examineService.sceneList(request);

        if(CollectionUtil.isNotEmpty(modelList)) {
            resultList = modelList.stream().map(v -> {
                ExamineSceneDto dto = BeanUtil.copyProperties(v, ExamineSceneDto.class);
                return dto;
            }).collect(Collectors.toList());
        }
        return resultList;
    }


    /************************************************** 对练QA **************************************************/

    /**
     * 更新QA
     * @param id
     * @param status
     */
    public void updateQaStatus(Integer id, Integer status) {
        examineService.updateExamineQaStatus(id, status);
    }

    /**
     * 删除QA
     * @param id
     */
    public void deleteQa(Integer id) {
        examineService.deleteExamineQa(id);
    }

    /**
     * 获取QA详情
     * @param id
     * @return
     */
    public ExamineQaDto qaDetail(Integer id) {
        ExamineQaResponseModel model = examineService.examineQaDetail(id);

        ExamineSceneCategoryDto category = sceneCategoryDetail(model.getCategoryId());
        if(Objects.nonNull(category)) {
            model.setCategoryName(category.getName());
        }

        return BeanUtil.copyProperties(model, ExamineQaDto.class);
    }


    /**
     * 保存QA
     * @param dto
     */
    public void saveExamineQa(ExamineQaSaveDto dto) {

        Assert.notNull(dto.getCategoryId(), "分类ID不能为空");

        ExamineSceneCategoryResponseModel category = examineService.sceneCategoryDetail(dto.getCategoryId());
        Assert.notNull(category, "分类信息获取异常");

        ExamineQaSaveModel model = BeanUtil.copyProperties(dto, ExamineQaSaveModel.class);
        examineService.saveExamineQa(model);
    }


    /**
     * QA分页查询
     * @param request
     * @return
     */
    public PageResult<ExamineQaDto> examineQaPaginQuery(ExamineQaRequestDto request) {

        ExamineQaRequestModel requestModel = BeanUtil.copyProperties(request, ExamineQaRequestModel.class);

        Page<ExamineQaResponseModel> page = examineService.examineQaPaginQuery(requestModel);

        PageResult<ExamineQaDto> pageResult = PageResult.of(request.getPageNo(), request.getPageSize(), (int) page.getTotalRow(), null);

        if(CollectionUtil.isNotEmpty(page.getRecords())) {
            List<ExamineQaDto> dtoList = page.getRecords().stream().map(v -> {
                ExamineQaDto dto = BeanUtil.copyProperties(v, ExamineQaDto.class);

                ExamineSceneCategoryDto category = sceneCategoryDetail(dto.getCategoryId());
                if(Objects.nonNull(category)) {
                    dto.setCategoryName(category.getName());
                }
                return dto;
            }).collect(Collectors.toList());

            pageResult.setItems(dtoList);
        }
        return pageResult;
    }


    /**
     * QA列表查询
     * @param examineId
     * @return
     */
    public List<ExamineQaDto> examineQaList(Integer examineId) {

        List<ExamineQaDto> resultList = new ArrayList<>();

        List<ExamineQaResponseModel> modelList = examineService.examineQaList(examineId);

        if(CollectionUtil.isNotEmpty(modelList)) {
            resultList = modelList.stream().map(v -> {
                ExamineQaDto dto = BeanUtil.copyProperties(v, ExamineQaDto.class);
                return dto;
            }).collect(Collectors.toList());
        }
        return resultList;
    }


    /************************************************** 对练 **************************************************/

    /**
     * 更新对练
     * @param id
     * @param status
     */
    public void updateExamineStatus(Integer id, Integer status) {
        examineService.updateExamineStatus(id, status);
    }

    /**
     * 删除对练
     * @param id
     */
    public void deleteExamine(Integer id) {
        examineService.deleteExamine(id);
    }

    /**
     * 获取对练详情
     * @param id
     * @return
     */
    public ExamineDto examineDetail(Integer id) {
        ExamineResponseModel model = examineService.examineDetail(id);
        // 获取关联题目信息
        List<ExamineQaXrefResponseModel> xrefList = examineService.examineQaXrefList(id);
        if(CollectionUtil.isNotEmpty(xrefList)) {
            List<Integer> qaIdList = xrefList.stream().map(ExamineQaXrefResponseModel::getQaId).collect(Collectors.toList());
            List<ExamineQaResponseModel> examineQaList = examineService.examineQaList(qaIdList);
            model.setQaList(examineQaList);
        }
        return BeanUtil.copyProperties(model, ExamineDto.class);
    }

    /**
     * 获取对练详情
     * @param id
     * @return
     */
    public ExamineMobileDto examineMobileDetail(Integer id) {
        ExamineMobileDto result = null;
        ExamineResponseModel model = examineService.examineDetail(id);
        if(Objects.nonNull(model)) {
            result = BeanUtil.copyProperties(model, ExamineMobileDto.class);

            // 考试次数
            long examTotalCount = 0;
            long examCount = 0;
            long practiceCount = 0;
            double maxOverallScore = 0;
            ExamineRecordRequestModel req = new ExamineRecordRequestModel();
            req.setExamineId(id);
            List<ExamineRecordResponseModel> recordList = examineService.examineRecordList(req);
            if(CollectionUtil.isNotEmpty(recordList)) {
                examCount = recordList.stream().filter(v -> Objects.equals(v.getExamineMode(), 1)).count();
                examTotalCount = recordList.size();
                practiceCount = examTotalCount - examTotalCount;

                maxOverallScore = recordList.stream().filter(v -> Objects.nonNull(v.getOverallScore())).mapToDouble(ExamineRecordResponseModel::getOverallScore).max().getAsDouble();
            }
            result.setExamCount(examCount);
            result.setPracticeCount(practiceCount);
            result.setExamTotalCount(examTotalCount);
            result.setMaxOverallScore(maxOverallScore);
        }
        return result;
    }

    /**
     * 获取对练记录详情
     * @param id
     * @return
     */
    public ExamineRecordDto examineRecordDetail(Integer id) {
        ExamineRecordDto result = null;
        ExamineRecordResponseModel model = examineService.examineRecordDetail(id);
        if(Objects.nonNull(model)) {
            result = BeanUtil.copyProperties(model, ExamineRecordDto.class);

            ExamineResponseModel examine = examineService.examineDetail(result.getExamineId());
            if(Objects.nonNull(examine)) {
                result.setExamineName(examine.getName());
                result.setQaNum(examine.getQaNum());
            }

            // 对练明细
            ExamineDetailRecordRequestModel req = new ExamineDetailRecordRequestModel();
            req.setRecordId(id);
            List<ExamineDetailRecordResponseModel> detailRecordList = examineService.examineDetailRecordList(req);
            if(CollectionUtil.isNotEmpty(detailRecordList)) {
                // 题目信息
                List<Integer> qaIdList = detailRecordList.stream().map(ExamineDetailRecordResponseModel::getQaId).collect(Collectors.toList());
                List<ExamineQaResponseModel> examineQaList = examineService.examineQaList(qaIdList);
                Map<Integer, ExamineQaResponseModel> qaMap = examineQaList.stream().collect(toMap(ExamineQaResponseModel::getId, Function.identity()));

                List<ExamineDetailRecordMobileDto> details = detailRecordList.stream().map(v -> {
                    ExamineDetailRecordMobileDto detail = BeanUtil.copyProperties(v, ExamineDetailRecordMobileDto.class);

                    ExamineQaResponseModel qaModel = qaMap.get(detail.getQaId());
                    if(Objects.nonNull(qaModel)) {
                        detail.setQuestion(qaModel.getQuestion());
                        detail.setAnswer(qaModel.getAnswer());
                        detail.setUserAnswer(v.getAnswer());
                    }
                    return detail;
                }).collect(Collectors.toList());
                result.setDetails(details);
            }
        }
        return result;
    }

    /**
     * 获取对练记录详情
     * @param id
     * @return
     */
    public ExamineRecordMobileDto examineRecordMobileDetail(Integer id) {
        ExamineRecordMobileDto result = null;
        ExamineRecordResponseModel model = examineService.examineRecordDetail(id);
        if(Objects.nonNull(model)) {
            result = BeanUtil.copyProperties(model, ExamineRecordMobileDto.class);

            // 对练明细
            ExamineDetailRecordRequestModel req = new ExamineDetailRecordRequestModel();
            req.setRecordId(id);
            List<ExamineDetailRecordResponseModel> detailRecordList = examineService.examineDetailRecordList(req);
            if(CollectionUtil.isNotEmpty(detailRecordList)) {
                double totalScore = detailRecordList.stream().filter(v -> Objects.nonNull(v.getScore())).mapToDouble(ExamineDetailRecordResponseModel::getScore).sum();
                Double avgScore = totalScore / detailRecordList.size();
                // 题目信息
                List<Integer> qaIdList = detailRecordList.stream().map(ExamineDetailRecordResponseModel::getQaId).collect(Collectors.toList());
                List<ExamineQaResponseModel> examineQaList = examineService.examineQaList(qaIdList);
                Map<Integer, ExamineQaResponseModel> qaMap = examineQaList.stream().collect(toMap(ExamineQaResponseModel::getId, Function.identity()));

                List<ExamineDetailRecordMobileDto> details = detailRecordList.stream().map(v -> {
                    ExamineDetailRecordMobileDto detail = BeanUtil.copyProperties(v, ExamineDetailRecordMobileDto.class);

                    ExamineQaResponseModel qaModel = qaMap.get(detail.getQaId());
                    if(Objects.nonNull(qaModel)) {
                        detail.setQuestion(qaModel.getQuestion());
                        detail.setAnswer(qaModel.getAnswer());
                        detail.setUserAnswer(v.getAnswer());
                    }
                    return detail;
                }).collect(Collectors.toList());
                result.setAvgScore(avgScore.floatValue());
                result.setDetails(details);
            }
        }
        return result;
    }

    /**
     * 保存对练
     * @param dto
     */
    @Transactional
    public void saveExamine(ExamineSaveDto dto) {

        Assert.notNull(dto.getSceneId(), "场景不能为空");
        Assert.notNull(dto.getCategoryId(), "场景分类不能为空");
        Assert.notEmpty(dto.getQaIdList(), "对练题目不能为空");


        ExamineSceneResponseModel scene = examineService.sceneDetail(dto.getSceneId());
        Assert.notNull(scene, "场景信息获取异常");

        ExamineSceneCategoryResponseModel category = examineService.sceneCategoryDetail(dto.getCategoryId());
        Assert.notNull(category, "场景分类获取异常");

        ExamineSaveModel model = BeanUtil.copyProperties(dto, ExamineSaveModel.class);
        model.setQaNum(dto.getQaIdList().size());
        examineService.saveExamine(model);

        // 删除关联信息
        examineService.deleteExamineQaXref(model.getId());

        // 保存题目关联
        examineService.saveExamineQaXref(model.getId(), dto.getQaIdList());
    }


    /**
     * 对练分页查询
     * @param request
     * @return
     */
    public PageResult<ExamineDto> examinePaginQuery(ExamineRequestDto request) {

        ExamineRequestModel requestModel = BeanUtil.copyProperties(request, ExamineRequestModel.class);

        Page<ExamineResponseModel> page = examineService.examinePaginQuery(requestModel);

        PageResult<ExamineDto> pageResult = PageResult.of(request.getPageNo(), request.getPageSize(), (int) page.getTotalRow(), null);

        if(CollectionUtil.isNotEmpty(page.getRecords())) {
            List<ExamineDto> dtoList = page.getRecords().stream().map(v -> {
                ExamineDto dto = BeanUtil.copyProperties(v, ExamineDto.class);

                dto.setStartTime(v.getStartTime().getTime());
                dto.setEndTime(v.getEndTime().getTime());

                return dto;
            }).collect(Collectors.toList());

            pageResult.setItems(dtoList);
        }
        return pageResult;
    }

    /**
     * 对练分页查询
     * @param request
     * @return
     */
    public PageResult<ExamineMobileDto> examineMobilePaginQuery(ExamineRequestDto request) {

        ExamineRequestModel requestModel = BeanUtil.copyProperties(request, ExamineRequestModel.class);
        requestModel.setStatus(1);

        Page<ExamineResponseModel> page = examineService.examinePaginQuery(requestModel);

        PageResult<ExamineMobileDto> pageResult = PageResult.of(request.getPageNo(), request.getPageSize(), (int) page.getTotalRow(), null);

        if(CollectionUtil.isNotEmpty(page.getRecords())) {
            List<ExamineMobileDto> dtoList = page.getRecords().stream().map(v -> {
                ExamineMobileDto dto = BeanUtil.copyProperties(v, ExamineMobileDto.class);

                // 已考人数
                long examCount = 0;
                ExamineRecordRequestModel req = new ExamineRecordRequestModel();
                req.setExamineId(v.getId());
                List<ExamineRecordResponseModel> recordList = examineService.examineRecordList(req);
                if(CollectionUtil.isNotEmpty(recordList)) {
                    examCount = recordList.size();
                }
                dto.setExamTotalCount(examCount);

                return dto;
            }).collect(Collectors.toList());

            pageResult.setItems(dtoList);
        }
        return pageResult;
    }

    /**
     * 对练列表查询
     * @param request
     * @return
     */
    public List<ExamineDto> examineList(ExamineRequestModel request) {

        List<ExamineDto> resultList = new ArrayList<>();

        List<ExamineResponseModel> modelList = examineService.examineList(request);

        if(CollectionUtil.isNotEmpty(modelList)) {
            resultList = modelList.stream().map(v -> {
                ExamineDto dto = BeanUtil.copyProperties(v, ExamineDto.class);
                return dto;
            }).collect(Collectors.toList());
        }
        return resultList;
    }

    /**
     * 用户对练记录
     * @param request
     * @return
     */
    public PageResult<ExamineRecordMobileDto> examineRecordMobilePaginQuery(ExamineRecordMobileRequestDto request) {

        ExamineRecordRequestModel requestModel = BeanUtil.copyProperties(request, ExamineRecordRequestModel.class);

        Page<ExamineRecordResponseModel> page = examineService.examineRecordPaginQuery(requestModel);

        PageResult<ExamineRecordMobileDto> pageResult = PageResult.of(request.getPageNo(), request.getPageSize(), (int) page.getTotalRow(), null);

        if(CollectionUtil.isNotEmpty(page.getRecords())) {
            List<ExamineRecordMobileDto> dtoList = page.getRecords().stream().map(v -> {
                ExamineRecordMobileDto dto = BeanUtil.copyProperties(v, ExamineRecordMobileDto.class);

                ExamineDto examineDto = examineDetail(dto.getExamineId());
                if(Objects.nonNull(examineDto)) {
                    dto.setExamineName(examineDto.getName());
                    dto.setExamineDescription(examineDto.getDescription());
                }

                return dto;
            }).collect(Collectors.toList());

            pageResult.setItems(dtoList);
        }
        return pageResult;
    }


    /**
     * 用户对练记录
     * @param request
     * @return
     */
    public PageResult<ExamineRecordDto> examineRecordPaginQuery(ExamineRecordRequestDto request) {

        ExamineRecordRequestModel requestModel = BeanUtil.copyProperties(request, ExamineRecordRequestModel.class);

        Page<ExamineRecordResponseModel> page = examineService.examineRecordPaginQuery(requestModel);

        PageResult<ExamineRecordDto> pageResult = PageResult.of(request.getPageNo(), request.getPageSize(), (int) page.getTotalRow(), null);

        if(CollectionUtil.isNotEmpty(page.getRecords())) {
            List<ExamineRecordDto> dtoList = page.getRecords().stream().map(v -> {
                ExamineRecordDto dto = BeanUtil.copyProperties(v, ExamineRecordDto.class);

                ExamineDto examineDto = examineDetail(dto.getExamineId());
                if(Objects.nonNull(examineDto)) {
                    dto.setExamineName(examineDto.getName());
                    dto.setExamineDescription(examineDto.getDescription());
                    dto.setQaNum(examineDto.getQaNum());
                }

                return dto;
            }).collect(Collectors.toList());

            pageResult.setItems(dtoList);
        }
        return pageResult;
    }


    /************************************************** 资料库分类 **************************************************/

    /**
     * 更新资料库分类
     * @param id
     * @param status
     */
    public void updateWikiCategoryStatus(Integer id, Integer status) {
        examineService.updateWikiCategoryStatus(id, status);
    }

    /**
     * 删除资料库分类
     * @param id
     */
    public void deleteWikiCategory(Integer id) {
        examineService.deleteWikiCategory(id);
    }

    /**
     * 获取资料库分类详情
     * @param id
     * @return
     */
    public WikiCategoryDto wikiCategoryDetail(Integer id) {
        WikiCategoryResponseModel model = examineService.wikiCategoryDetail(id);
        return BeanUtil.copyProperties(model, WikiCategoryDto.class);
    }


    /**
     * 保存资料库分类
     * @param dto
     */
    public void saveWikiCategory(WikiCategorySaveDto dto) {

        Assert.notNull(dto.getParentId(), "分类父级ID不能为空");

        WikiCategorySaveModel model = BeanUtil.copyProperties(dto, WikiCategorySaveModel.class);
        examineService.saveWikiCategory(model);
    }


    /**
     * 资料库分类分页查询
     * @param request
     * @return
     */
    public PageResult<WikiCategoryDto> wikiCategoryPaginQuery(WikiCategoryRequestDto request) {

        WikiCategoryRequestModel requestModel = BeanUtil.copyProperties(request, WikiCategoryRequestModel.class);

        Page<WikiCategoryResponseModel> page = examineService.wikiCategoryPaginQuery(requestModel);

        PageResult<WikiCategoryDto> pageResult = PageResult.of(request.getPageNo(), request.getPageSize(), (int) page.getTotalRow(), null);

        if(CollectionUtil.isNotEmpty(page.getRecords())) {
            List<WikiCategoryDto> dtoList = page.getRecords().stream().map(v -> {
                WikiCategoryDto dto = BeanUtil.copyProperties(v, WikiCategoryDto.class);
                return dto;
            }).collect(Collectors.toList());

            pageResult.setItems(dtoList);
        }
        return pageResult;
    }

    /**
     * 资料库分类分页查询
     * @param request
     * @return
     */
    public PageResult<WikiCategoryMobileDto> wikiCategoryMobilePaginQuery(WikiCategoryRequestDto request) {

        WikiCategoryRequestModel requestModel = BeanUtil.copyProperties(request, WikiCategoryRequestModel.class);

        Page<WikiCategoryResponseModel> page = examineService.wikiCategoryPaginQuery(requestModel);

        PageResult<WikiCategoryMobileDto> pageResult = PageResult.of(request.getPageNo(), request.getPageSize(), (int) page.getTotalRow(), null);

        if(CollectionUtil.isNotEmpty(page.getRecords())) {
            List<WikiCategoryMobileDto> dtoList = page.getRecords().stream().map(v -> {
                WikiCategoryMobileDto dto = BeanUtil.copyProperties(v, WikiCategoryMobileDto.class);
                return dto;
            }).collect(Collectors.toList());

            pageResult.setItems(dtoList);
        }
        return pageResult;
    }


    /**
     * 资料库分类列表查询
     * @param request
     * @return
     */
    public List<WikiCategoryDto> wikiCategoryList(WikiCategoryRequestDto request) {

        List<WikiCategoryDto> resultList = new ArrayList<>();

        WikiCategoryRequestModel requestModel = BeanUtil.copyProperties(request, WikiCategoryRequestModel.class);

        List<WikiCategoryResponseModel> modelList = examineService.wikiCategoryList(requestModel);

        if(CollectionUtil.isNotEmpty(modelList)) {
            resultList = modelList.stream().map(v -> {
                WikiCategoryDto dto = BeanUtil.copyProperties(v, WikiCategoryDto.class);
                return dto;
            }).collect(Collectors.toList());
        }
        return resultList;
    }


    /************************************************** 资料库 **************************************************/

    /**
     * 更新资料库
     * @param id
     * @param status
     */
    public void updateWikiStatus(Integer id, Integer status) {
        examineService.updateWikiStatus(id, status);
    }

    /**
     * 删除资料库
     * @param id
     */
    public void deleteWiki(Integer id) {
        examineService.deleteWiki(id);
    }

    /**
     * 获取资料库详情
     * @param id
     * @return
     */
    public WikiDto wikiDetail(Integer id) {
        WikiResponseModel model = examineService.wikiDetail(id);
        return BeanUtil.copyProperties(model, WikiDto.class);
    }

    public WikiMobileDto wikiMobileDetail(Integer id) {
        WikiResponseModel model = examineService.wikiDetail(id);
        return BeanUtil.copyProperties(model, WikiMobileDto.class);
    }

    /**
     * 保存资料库
     * @param dto
     */
    public void saveWiki(WikiSaveDto dto) {

        WikiSaveModel model = BeanUtil.copyProperties(dto, WikiSaveModel.class);
        examineService.saveWiki(model);
    }


    /**
     * 资料库分页查询
     * @param request
     * @return
     */
    public PageResult<WikiDto> wikiPaginQuery(WikiRequestDto request) {

        WikiRequestModel requestModel = BeanUtil.copyProperties(request, WikiRequestModel.class);

        Page<WikiResponseModel> page = examineService.wikiPaginQuery(requestModel);

        PageResult<WikiDto> pageResult = PageResult.of(request.getPageNo(), request.getPageSize(), (int) page.getTotalRow(), null);

        if(CollectionUtil.isNotEmpty(page.getRecords())) {
            List<WikiDto> dtoList = page.getRecords().stream().map(v -> {
                WikiDto dto = BeanUtil.copyProperties(v, WikiDto.class);

                ExamineSceneCategoryDto sceneCategoryDto = sceneCategoryDetail(dto.getSceneCategoryId());
                if(Objects.nonNull(sceneCategoryDto)) {
                    dto.setSceneCategoryName(sceneCategoryDto.getName());
                }

                WikiCategoryDto wikiCategoryDto = wikiCategoryDetail(dto.getWikiCategoryId());
                if(Objects.nonNull(wikiCategoryDto)) {
                    dto.setWikiCategoryName(wikiCategoryDto.getName());
                }

                return dto;
            }).collect(Collectors.toList());

            pageResult.setItems(dtoList);
        }
        return pageResult;
    }


    /**
     * 资料库分页查询
     * @param request
     * @return
     */
    public PageResult<WikiListMobileDto> wikiMobilePaginQuery(WikiRequestDto request) {

        WikiRequestModel requestModel = BeanUtil.copyProperties(request, WikiRequestModel.class);

        Page<WikiResponseModel> page = examineService.wikiPaginQuery(requestModel);

        PageResult<WikiListMobileDto> pageResult = PageResult.of(request.getPageNo(), request.getPageSize(), (int) page.getTotalRow(), null);

        if(CollectionUtil.isNotEmpty(page.getRecords())) {
            List<WikiListMobileDto> dtoList = page.getRecords().stream().map(v -> {
                WikiListMobileDto dto = BeanUtil.copyProperties(v, WikiListMobileDto.class);

                WikiCategoryDto wikiCategoryDto = wikiCategoryDetail(dto.getWikiCategoryId());
                if(Objects.nonNull(wikiCategoryDto)) {
                    dto.setWikiCategoryName(wikiCategoryDto.getName());
                }

                return dto;
            }).collect(Collectors.toList());

            pageResult.setItems(dtoList);
        }
        return pageResult;
    }

    /**
     * 资料库列表查询
     * @param request
     * @return
     */
    public List<WikiDto> wikiList(WikiRequestDto request) {

        List<WikiDto> resultList = new ArrayList<>();

        WikiRequestModel requestModel = BeanUtil.copyProperties(request, WikiRequestModel.class);

        List<WikiResponseModel> modelList = examineService.wikiList(requestModel);

        if(CollectionUtil.isNotEmpty(modelList)) {
            resultList = modelList.stream().map(v -> {
                WikiDto dto = BeanUtil.copyProperties(v, WikiDto.class);
                return dto;
            }).collect(Collectors.toList());
        }
        return resultList;
    }

    /************************************************** 移动端对练 **************************************************/


    /**
     * 开始考试，并返回考试流水号&第一道题目
     * @param userPrincipal
     * @param request
     * @return
     */
    public ExamineStartResultDto startExamine(UserPrincipal userPrincipal, StartExamineMobileRequestDto request) {

        String userId = userPrincipal.getUserId();

        Integer examineId = request.getExamineId();
        Integer examineMode = request.getExamineMode();
        String businessNo = UUID.randomUUID().toString();

        ExamineStartResultDto result = new ExamineStartResultDto();
        result.setBusinessNo(businessNo);

        // 获取缓存考试会话信息
        String examBusinessKey = String.format(EXAM_BUSINESS_REDIS_KEY, businessNo);
        ExamineBusinessCacheDto businessCache = getExamBusinessCache(businessNo);
        if(Objects.isNull(businessCache)) {

            businessCache = new ExamineBusinessCacheDto();
            businessCache.setExamineId(examineId);
            businessCache.setExamineMode(examineMode);
            businessCache.setUserId(userId);
        }

        // 当前出题
        Integer currentQuestionId = businessCache.getCurrentQuestionId();
        if(Objects.isNull(currentQuestionId)) {
            List<ExamineQaResponseModel> examineQaList = examineService.examineQaList(examineId);
            if(CollectionUtil.isNotEmpty(examineQaList)) {
                ExamineQaResponseModel model = examineQaList.get(0);
                currentQuestionId = model.getId();

                result.setQuestionId(currentQuestionId);
                result.setQuestion(model.getQuestion());
                result.setAnswer(model.getAnswer());
                result.setNext(examineQaList.size() > 1 ? true : false);

                businessCache.setCurrentQuestionId(currentQuestionId);
                businessCache.setHasNext(result.isNext());
            }
        }

        // 缓存
        redisTemplate.opsForValue().set(examBusinessKey, JSONObject.toJSONString(businessCache), 1, TimeUnit.DAYS);

        return result;
    }


    /**
     * 获取下一题
     * @param userPrincipal
     * @param examineId 对练ID
     * @param businessNo 考试流水号
     * @return
     */
    public ExamineStartResultDto getNextQuestion(UserPrincipal userPrincipal, Integer examineId, String businessNo) {

        String userId = userPrincipal.getUserId();

        ExamineStartResultDto result = new ExamineStartResultDto();

        result.setBusinessNo(businessNo);

        // 获取缓存考试会话信息
        String examBusinessKey = String.format(EXAM_BUSINESS_REDIS_KEY, businessNo);
        ExamineBusinessCacheDto businessCache = getExamBusinessCache(businessNo);
        if(Objects.isNull(businessCache)) {
            throw new InternalException(-1, "获取不到考试流水信息");
        }
        // 当前出题
        Integer currentQuestionId = businessCache.getCurrentQuestionId();
        if(Objects.isNull(currentQuestionId)) {
            throw new InternalException(-1, "获取上一题信息失败");
        }
        // 获取下一题
        ExamineQaResponseModel nextQa = null;
        boolean hasNext = false;
        List<ExamineQaResponseModel> examineQaList = examineService.examineQaList(examineId);
        if(CollectionUtil.isNotEmpty(examineQaList)) {
            OptionalInt indexOpt = IntStream.range(0, examineQaList.size())
                    .filter(i -> examineQaList.get(i).getId() == currentQuestionId)
                    .findFirst();
            if (indexOpt.isPresent()) {
                int index = indexOpt.getAsInt();
                nextQa = index + 1 < examineQaList.size() ? examineQaList.get(index + 1) : null;
                int remainingNum = examineQaList.size() - index - 1;
                hasNext = remainingNum > 1 ? true : false;
            }
        }
        if(Objects.nonNull(nextQa)) {
            // 返回结果
            result.setNext(hasNext);
            result.setQuestion(nextQa.getQuestion());
            result.setAnswer(nextQa.getAnswer());
            result.setQuestionId(nextQa.getId());

            // 缓存
            businessCache.setCurrentQuestionId(nextQa.getId());
            businessCache.setHasNext(hasNext);
            redisTemplate.opsForValue().set(examBusinessKey, JSONObject.toJSONString(businessCache), 1, TimeUnit.DAYS);

            return result;
        } else {
            throw new InternalException(-1, "获取下一题信息有误");
        }
    }


    /**
     * 考试答题
     * @param userPrincipal
     * @param request
     */
    public void submitAnswer(UserPrincipal userPrincipal, SubmitAnswerMobileRequestDto request) {

        String userId = userPrincipal.getUserId();
        Integer examineId = request.getExamineId();
        String answer = request.getAnswer();
        Integer questionId = request.getQuestionId();
        String businessNo = request.getBusinessNo();

        ExamineStartResultDto result = new ExamineStartResultDto();
        result.setBusinessNo(businessNo);

        // 获取缓存考试会话信息
        String examBusinessKey = String.format(EXAM_BUSINESS_REDIS_KEY, businessNo);
        ExamineBusinessCacheDto businessCache = getExamBusinessCache(businessNo);
        if(Objects.isNull(businessCache)) {
            throw new InternalException(-1, "获取不到对练信息");
        }

        // 获取题目信息
        ExamineQaDto examineQaDto = qaDetail(questionId);

        // 获取数据库考试信息
        Integer recordId = null;
        ExamineRecordResponseModel examineRecord = examineService.examineRecordDetail(businessNo);
        if(Objects.isNull(examineRecord)) {
            // 第一次答题，保存考试信息
            ExamineRecordSaveModel saveModel = new ExamineRecordSaveModel();
            saveModel.setUserId(userId);
            saveModel.setUserName(userPrincipal.getUserName());
            saveModel.setShopId(userPrincipal.getShopId());
            saveModel.setShopName(userPrincipal.getShopName());
            saveModel.setAnsweredNum(1);
            saveModel.setExamineId(examineId);
            saveModel.setExamineMode(businessCache.getExamineMode());
            saveModel.setBusinessNo(businessNo);
            examineService.saveExamineRecord(saveModel);

            recordId = saveModel.getId();
        } else {
            recordId = examineRecord.getId();

            // 更新答题数量
            ExamineRecordSaveModel saveModel = new ExamineRecordSaveModel();
            saveModel.setId(recordId);
            saveModel.setAnsweredNum(examineRecord.getAnsweredNum() + 1);
            examineService.saveExamineRecord(saveModel);
        }

        // 保存单题答题信息
        ExamineRecordDetailSaveModel detailSaveModel = new ExamineRecordDetailSaveModel();
        detailSaveModel.setRecordId(recordId);
        detailSaveModel.setQaId(questionId);
        detailSaveModel.setCreateTime(new Date());
        detailSaveModel.setAnswer(answer);

        // AI 点评
//        AiSingleEvaluateResultDto evaluateResultDto = extensionService.run4SingleEvaluate(businessNo, userId, examineQaDto.getQuestion(), examineQaDto.getAnswer(), answer);
//        if(Objects.nonNull(evaluateResultDto)) {
//            detailSaveModel.setScore(evaluateResultDto.getScore());
//            detailSaveModel.setEvaluation(evaluateResultDto.getEvaluation());
//        }
        examineService.saveExamineRecordDetail(detailSaveModel);

        // 定时任务
        ExamineEvaluateJobResponseModel evaluateJob = commonService.getEvaluateJob(0, detailSaveModel.getId());
        if(Objects.isNull(evaluateJob)) {
            ExamineEvaluateJobSaveModel model = new ExamineEvaluateJobSaveModel();
            model.setBusinessId(detailSaveModel.getId());
            model.setBusinessNo(businessNo);
            model.setType(0);
            model.setStatus(0);
            model.setUserId(userId);
            model.setCreateTime(new Date());
            commonService.saveEvaluateJob(model);
        }

        // 更新缓存
        businessCache.setRecordId(recordId);
        redisTemplate.opsForValue().set(examBusinessKey, JSONObject.toJSONString(businessCache), 1, TimeUnit.DAYS);

        // 最后一题答题，完成考试，执行AI总点评
        if(!businessCache.isHasNext()) {
            this.completeExamine(userPrincipal, examineId, businessNo);
        }
    }


    /**
     * 完成考试，更新最终考试结果
     * @param userPrincipal
     * @param examineId
     * @param businessNo
     */
    public void completeExamine(UserPrincipal userPrincipal, Integer examineId, String businessNo) {

        String userId = userPrincipal.getUserId();

        // 获取缓存考试会话信息
        ExamineBusinessCacheDto businessCache = getExamBusinessCache(businessNo);
        if(Objects.isNull(businessCache)) {
            throw new InternalException(-1, "获取不到对练信息");
        }
        Integer recordId = businessCache.getRecordId();

        ExamineEvaluateJobResponseModel evaluateJob = commonService.getEvaluateJob(1, recordId);
        if(Objects.isNull(evaluateJob)) {
            ExamineEvaluateJobSaveModel model = new ExamineEvaluateJobSaveModel();
            model.setBusinessId(recordId);
            model.setBusinessNo(businessNo);
            model.setType(1);
            model.setStatus(0);
            model.setUserId(userId);
            model.setCreateTime(new Date());
            commonService.saveEvaluateJob(model);
        }

        // 获取答题结果
//        ExamineDetailRecordRequestModel requestModel = new ExamineDetailRecordRequestModel();
//        requestModel.setRecordId(recordId);
//        List<ExamineDetailRecordResponseModel> detailRecordList = examineService.examineDetailRecordList(requestModel);
//        if(CollectionUtil.isNotEmpty(detailRecordList)) {
//            // 计算综合评分
//            double totalScore = detailRecordList.stream().filter(v -> Objects.nonNull(v.getScore())).mapToDouble(ExamineDetailRecordResponseModel::getScore).sum();
//            Double avgScore = totalScore / detailRecordList.size();
//
//            List<Integer> qaIdList = detailRecordList.stream().map(ExamineDetailRecordResponseModel::getQaId).collect(Collectors.toList());
//            List<ExamineQaResponseModel> examineQaList = examineService.examineQaList(qaIdList);
//            Map<Integer, ExamineQaResponseModel> qaMap = examineQaList.stream().collect(toMap(ExamineQaResponseModel::getId, Function.identity()));
//
//            String qaResultStr = "";
//            for(int i = 0; i < detailRecordList.size(); i++) {
//                ExamineDetailRecordResponseModel detail = detailRecordList.get(i);
//                ExamineQaResponseModel qa = qaMap.get(detail.getQaId());
//
//                qaResultStr += "问题 " + (i + 1) + ": " + qa.getQuestion() + "\n";
//                qaResultStr += "得分：" + detail.getScore() + "\n";
//                qaResultStr += "评分说明：" + detail.getEvaluation() + "\n";
//                qaResultStr += "----------------------------------------\n";
//            }
//
//            // 执行综合点评
//            String overallEvaluation = extensionService.run4TotalEvaluate(businessNo, userId, avgScore.floatValue(), qaResultStr);
//            // 更新答题数量
//            ExamineRecordSaveModel saveModel = new ExamineRecordSaveModel();
//            saveModel.setId(recordId);
//            saveModel.setOverallEvaluation(overallEvaluation);
//            saveModel.setOverallScore(avgScore.floatValue());
//
//            examineService.saveExamineRecord(saveModel);
//        }
    }


    /**
     * 获取考试最终结果
     * @param userPrincipal
     * @param examineId 对练ID
     * @param businessNo 考试流水号
     * @return
     */
    public ExamineRecordMobileDto getExamineFinalResult(UserPrincipal userPrincipal, Integer examineId, String businessNo) {
        // 获取考试记录信息
        String totalJobKey = String.format(Codes.TOTAL_AI_EVALUATE_JOB_STATUS, businessNo);
        String totalJobStatus = redisTemplate.opsForValue().get(totalJobKey);
        if(Objects.equals(totalJobStatus, "1")) {
            ExamineRecordResponseModel recordModel = examineService.examineRecordDetail(businessNo);
            if(Objects.nonNull(recordModel)) {
                Integer recordId = recordModel.getId();
                return this.examineRecordMobileDetail(recordId);
            }
        }
        return null;
    }

    /**
     * 获取考试详情
     * @param businessNo 考试流水号
     * @return
     */
    public ExamineRecordMobileDto getExamineRecordDetail(String businessNo) {
        // 获取考试记录信息
        ExamineRecordResponseModel recordModel = examineService.examineRecordDetail(businessNo);
        if(Objects.nonNull(recordModel)) {
            Integer recordId = recordModel.getId();
            return this.examineRecordMobileDetail(recordId);
        }
        return null;
    }

    /**
     * 退出考试，清除考试记录
     * @param userPrincipal
     * @param examineId
     * @param businessNo
     * @return
     */
    public void exitExamine(UserPrincipal userPrincipal, Integer examineId, String businessNo) {
        // 获取缓存考试会话信息
        String examBusinessKey = String.format(EXAM_BUSINESS_REDIS_KEY, businessNo);
        ExamineBusinessCacheDto businessCache = getExamBusinessCache(businessNo);

        if(Objects.nonNull(businessCache) && Objects.nonNull(businessCache.getRecordId())) {
            // 删除考试记录
            ExamineRecordSaveModel saveModel = new ExamineRecordSaveModel();
            saveModel.setId(businessCache.getRecordId());
            saveModel.setDeleted(1);
            examineService.saveExamineRecord(saveModel);
        }

        // 删除缓存
        redisTemplate.delete(examBusinessKey);
    }

    /**
     * 获取考试流水缓存信息
     * @param businessNo
     * @return
     */
    private ExamineBusinessCacheDto getExamBusinessCache(String businessNo) {
        // 获取缓存考试会话信息
        String examBusinessKey = String.format(EXAM_BUSINESS_REDIS_KEY, businessNo);
        String examBusinessCache = redisTemplate.opsForValue().get(examBusinessKey);
        if(StrUtil.isNotEmpty(examBusinessCache)) {
            return JSONObject.parseObject(examBusinessCache, ExamineBusinessCacheDto.class);
        }
        return null;
    }

}
