package cn.breeze.elleai.domain.sparring.service;

import cn.breeze.elleai.domain.sparring.model.request.*;
import cn.breeze.elleai.domain.sparring.model.response.*;
import cn.breeze.elleai.infra.entity.*;
import cn.breeze.elleai.infra.mapper.*;
import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.mybatisflex.core.paginate.Page;
import com.mybatisflex.core.query.QueryWrapper;
import com.mybatisflex.core.update.UpdateChain;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.apache.ibatis.annotations.Param;
import org.apache.ibatis.annotations.Select;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import static cn.breeze.elleai.infra.entity.table.ExamineQaTableDef.EXAMINE_QA_ENTITY;
import static cn.breeze.elleai.infra.entity.table.ExamineQaXrefTableDef.EXAMINE_QA_XREF_ENTITY;
import static cn.breeze.elleai.infra.entity.table.ExamineSceneCategoryTableDef.EXAMINE_SCENE_CATEGORY_ENTITY;
import static cn.breeze.elleai.infra.entity.table.ExamineSceneTableDef.EXAMINE_SCENE_ENTITY;
import static cn.breeze.elleai.infra.entity.table.ExamineTableDef.EXAMINE_ENTITY;
import static cn.breeze.elleai.infra.entity.table.UserExamineDetailRecordTableDef.USER_EXAMINE_DETAIL_RECORD_ENTITY;
import static cn.breeze.elleai.infra.entity.table.UserExamineRecordTableDef.USER_EXAMINE_RECORD_ENTITY;
import static cn.breeze.elleai.infra.entity.table.WikiCategoryTableDef.WIKI_CATEGORY_ENTITY;
import static cn.breeze.elleai.infra.entity.table.WikiTableDef.WIKI_ENTITY;

/**
 *  服务层实现。
 *
 * @author breeze
 * @since 2024-09-18
 */
@Service
@RequiredArgsConstructor
public class ExamineServiceImpl implements ExamineService {

    private final ExamineMapper examineMapper;

    private final ExamineQaMapper qaMapper;

    private final ExamineSceneMapper sceneMapper;

    private final ExamineSceneCategoryMapper sceneCategoryMapper;

    private final ExamineQaXrefMapper qaXrefMapper;

    private final ProperNounMapper properNounMapper;

    private final WikiMapper wikiMapper;

    private final WikiCategoryMapper wikiCategoryMapper;

    private final UserExamineRecordMapper userExamineRecordMapper;

    private final UserExamineDetailRecordMapper userExamineDetailRecordMapper;

    @Override
    public Page<ExamineSceneResponseModel> scenePaginQuery(ExamineSceneRequestModel request) {

        Integer pageNo = ObjectUtil.defaultIfNull(request.getPageNo(), 1);
        Integer pageSize = ObjectUtil.defaultIfNull(request.getPageSize(), 10);

        QueryWrapper queryWrapper = QueryWrapper.create();
        queryWrapper.select(EXAMINE_SCENE_ENTITY.ALL_COLUMNS);
        queryWrapper.from(EXAMINE_SCENE_ENTITY.getTableName());
        queryWrapper.where(EXAMINE_SCENE_ENTITY.DELETED.eq(0));
        if(StrUtil.isNotEmpty(request.getName())) {
            queryWrapper.where(EXAMINE_SCENE_ENTITY.NAME.like("%"+request.getName()+"%"));
        }
        if(Objects.nonNull(request.getStatus())) {
            queryWrapper.where(EXAMINE_SCENE_ENTITY.STATUS.eq(request.getStatus()));
        }

        Page<ExamineSceneResponseModel> page = sceneMapper.paginateAs(pageNo, pageSize, queryWrapper, ExamineSceneResponseModel.class);
        
        return page;
    }

    @Override
    public Page<ExamineSceneCategoryResponseModel> sceneCategoryPaginQuery(ExamineSceneCategoryRequestModel request) {
        Integer pageNo = ObjectUtil.defaultIfNull(request.getPageNo(), 1);
        Integer pageSize = ObjectUtil.defaultIfNull(request.getPageSize(), 10);

        QueryWrapper queryWrapper = QueryWrapper.create();
        queryWrapper.select(EXAMINE_SCENE_CATEGORY_ENTITY.ALL_COLUMNS);
        queryWrapper.from(EXAMINE_SCENE_CATEGORY_ENTITY.getTableName());
        queryWrapper.where(EXAMINE_SCENE_CATEGORY_ENTITY.DELETED.eq(0));
        if(StringUtils.isNotEmpty(request.getName())) {
            queryWrapper.where(EXAMINE_SCENE_CATEGORY_ENTITY.NAME.like("%"+request.getName()+"%"));
        }
        if(Objects.nonNull(request.getStatus())) {
            queryWrapper.where(EXAMINE_SCENE_CATEGORY_ENTITY.STATUS.eq(request.getStatus()));
        }
        if(Objects.nonNull(request.getSceneId())) {
            queryWrapper.where(EXAMINE_SCENE_CATEGORY_ENTITY.SCENE_ID.eq(request.getSceneId()));
        }
        Page<ExamineSceneCategoryResponseModel> page = sceneCategoryMapper.paginateAs(pageNo, pageSize, queryWrapper, ExamineSceneCategoryResponseModel.class);

        return page;
    }

    @Override
    public List<ExamineSceneResponseModel> sceneList(ExamineSceneRequestModel request) {
        QueryWrapper queryWrapper = QueryWrapper.create().where(EXAMINE_SCENE_ENTITY.DELETED.eq(0));
        if(StrUtil.isNotEmpty(request.getName())) {
            queryWrapper.where(EXAMINE_SCENE_ENTITY.NAME.like("%"+request.getName()+"%"));
        }
        return sceneMapper.selectListByQueryAs(queryWrapper, ExamineSceneResponseModel.class);
    }

    @Override
    public List<ExamineSceneCategoryResponseModel> sceneCategoryList(ExamineSceneCategoryRequestModel request) {
        QueryWrapper queryWrapper = QueryWrapper.create().where(EXAMINE_SCENE_CATEGORY_ENTITY.DELETED.eq(0));
        if(StrUtil.isNotEmpty(request.getName())) {
            queryWrapper.where(EXAMINE_SCENE_CATEGORY_ENTITY.NAME.like("%"+request.getName()+"%"));
        }
        return sceneCategoryMapper.selectListByQueryAs(queryWrapper, ExamineSceneCategoryResponseModel.class);
    }

    @Override
    public ExamineSceneResponseModel sceneDetail(Integer sceneId) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(EXAMINE_SCENE_ENTITY.ID.eq(sceneId).and(EXAMINE_SCENE_ENTITY.DELETED.eq(0)));
        return sceneMapper.selectOneByQueryAs(queryWrapper, ExamineSceneResponseModel.class);
    }

    @Override
    public ExamineSceneCategoryResponseModel sceneCategoryDetail(Integer sceneCategoryId) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(EXAMINE_SCENE_CATEGORY_ENTITY.ID.eq(sceneCategoryId).and(EXAMINE_SCENE_CATEGORY_ENTITY.DELETED.eq(0)));
        return sceneCategoryMapper.selectOneByQueryAs(queryWrapper, ExamineSceneCategoryResponseModel.class);
    }

    @Override
    public void updateSceneStatus(Integer sceneId, Integer status) {
        UpdateChain.of(ExamineSceneEntity.class)
                .set(EXAMINE_SCENE_ENTITY.STATUS, status)
                .set(EXAMINE_SCENE_ENTITY.UPDATE_TIME, new Date())
                .where(EXAMINE_SCENE_ENTITY.ID.eq(sceneId))
                .update();
    }

    @Override
    public void updateSceneCategoryStatus(Integer sceneCategoryId, Integer status) {
        UpdateChain.of(ExamineSceneCategoryEntity.class)
                .set(EXAMINE_SCENE_CATEGORY_ENTITY.STATUS, status)
                .set(EXAMINE_SCENE_CATEGORY_ENTITY.UPDATE_TIME, new Date())
                .where(EXAMINE_SCENE_CATEGORY_ENTITY.ID.eq(sceneCategoryId))
                .update();
    }

    @Override
    public void deleteScene(Integer sceneId) {
        UpdateChain.of(ExamineSceneEntity.class)
                .set(EXAMINE_SCENE_ENTITY.DELETED, 1)
                .set(EXAMINE_SCENE_ENTITY.UPDATE_TIME, new Date())
                .where(EXAMINE_SCENE_ENTITY.ID.eq(sceneId))
                .update();
    }

    @Override
    public void deleteSceneCategory(Integer sceneCategoryId) {
        UpdateChain.of(ExamineSceneCategoryEntity.class)
                .set(EXAMINE_SCENE_CATEGORY_ENTITY.DELETED, 1)
                .set(EXAMINE_SCENE_CATEGORY_ENTITY.UPDATE_TIME, new Date())
                .where(EXAMINE_SCENE_CATEGORY_ENTITY.ID.eq(sceneCategoryId))
                .update();
    }

    @Override
    public void saveScene(ExamineSceneSaveModel dto) {
        ExamineSceneEntity entity = BeanUtil.toBean(dto, ExamineSceneEntity.class);
        entity.setUpdateTime(new Date());
        if(Objects.isNull(entity.getId())) {
            entity.setStatus(0);
            entity.setDeleted(0);
        }
        sceneMapper.insertOrUpdateSelective(entity);
    }

    @Override
    public void saveSceneCategory(ExamineSceneCategorySaveModel dto) {
        ExamineSceneCategoryEntity entity = BeanUtil.toBean(dto, ExamineSceneCategoryEntity.class);
        entity.setUpdateTime(new Date());
        if(Objects.isNull(entity.getId())) {
            entity.setStatus(0);
            entity.setDeleted(0);
        }
        sceneCategoryMapper.insertOrUpdateSelective(entity);
    }

    @Override
    public Page<ExamineQaResponseModel> examineQaPaginQuery(ExamineQaRequestModel request) {
        Integer pageNo = ObjectUtil.defaultIfNull(request.getPageNo(), 1);
        Integer pageSize = ObjectUtil.defaultIfNull(request.getPageSize(), 10);

        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(EXAMINE_QA_ENTITY.DELETED.eq(0));
        if(StrUtil.isNotEmpty(request.getName())) {
            queryWrapper.where(EXAMINE_QA_ENTITY.QUESTION.like("%"+request.getName()+"%"));
        }

        Page<ExamineQaResponseModel> page = qaMapper.paginateAs(pageNo, pageSize, queryWrapper, ExamineQaResponseModel.class);

        return page;
    }

    @Override
    public List<ExamineQaResponseModel> examineQaList(Integer examineId) {
        QueryWrapper queryWrapper = QueryWrapper.create();
        queryWrapper.select(EXAMINE_QA_ENTITY.ALL_COLUMNS).from(EXAMINE_QA_ENTITY);
        queryWrapper.leftJoin(EXAMINE_QA_XREF_ENTITY).on(EXAMINE_QA_ENTITY.ID.eq(EXAMINE_QA_XREF_ENTITY.QA_ID))
                .and(EXAMINE_QA_XREF_ENTITY.DELETED.eq(0));
        queryWrapper.where(EXAMINE_QA_ENTITY.DELETED.eq(0));
        queryWrapper.where(EXAMINE_QA_XREF_ENTITY.EXAMINE_ID.eq(examineId));
        queryWrapper.orderBy(EXAMINE_QA_XREF_ENTITY.CREATE_TIME, false);

        System.out.println(queryWrapper.toSQL());
        return qaMapper.selectListByQueryAs(queryWrapper, ExamineQaResponseModel.class);
    }

    @Override
    public List<ExamineQaResponseModel> examineQaList(List<Integer> qaIds) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(EXAMINE_QA_ENTITY.ID.in(qaIds).and(EXAMINE_QA_ENTITY.DELETED.eq(0)));
        return qaMapper.selectListByQueryAs(queryWrapper, ExamineQaResponseModel.class);
    }

    @Override
    public ExamineQaResponseModel examineQaDetail(Integer qaId) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(EXAMINE_QA_ENTITY.ID.eq(qaId).and(EXAMINE_QA_ENTITY.DELETED.eq(0)));
        return qaMapper.selectOneByQueryAs(queryWrapper, ExamineQaResponseModel.class);
    }

    private void xx() {
//        @Select(value = "select max(question_id) from eleh_session_qa_record where tenant = #{tenant} and type = 0 and session_id = #{sessionId} and user_id = #{userId}")
//        Long selectSessionMaxQuestionId(@Param("examine_id") Integer examineId, @Param("sessionId") String sessionId, @Param("userId") String userId);

    }

    @Override
    public void updateExamineQaStatus(Integer qaId, Integer status) {
        UpdateChain.of(ExamineQaEntity.class)
                .set(EXAMINE_QA_ENTITY.STATUS, status)
                .set(EXAMINE_QA_ENTITY.UPDATE_TIME, new Date())
                .where(EXAMINE_QA_ENTITY.ID.eq(qaId))
                .update();
    }

    @Override
    public void deleteExamineQa(Integer qaId) {
        UpdateChain.of(ExamineQaEntity.class)
                .set(EXAMINE_QA_ENTITY.DELETED, 1)
                .set(EXAMINE_QA_ENTITY.UPDATE_TIME, new Date())
                .where(EXAMINE_QA_ENTITY.ID.eq(qaId))
                .update();
    }

    @Override
    public void saveExamineQa(ExamineQaSaveModel dto) {

        ExamineQaEntity entity = BeanUtil.toBean(dto, ExamineQaEntity.class);
        entity.setUpdateTime(new Date());
        if(Objects.isNull(entity.getId())) {
            entity.setStatus(0);
            entity.setDeleted(0);
        }
        qaMapper.insertOrUpdateSelective(entity);
    }

    @Override
    public Page<ExamineResponseModel> examinePaginQuery(ExamineRequestModel request) {
        Integer pageNo = ObjectUtil.defaultIfNull(request.getPageNo(), 1);
        Integer pageSize = ObjectUtil.defaultIfNull(request.getPageSize(), 10);

        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(EXAMINE_ENTITY.DELETED.eq(0));
        if(StrUtil.isNotEmpty(request.getName())) {
            queryWrapper.where(EXAMINE_ENTITY.NAME.like("%"+request.getName()+"%"));
        }
        if(Objects.nonNull(request.getStatus())) {
            queryWrapper.where(EXAMINE_ENTITY.STATUS.eq(request.getStatus()));
        }
        if(Objects.nonNull(request.getCategoryId())) {
            queryWrapper.where(EXAMINE_ENTITY.CATEGORY_ID.eq(request.getCategoryId()));
        }

        Page<ExamineResponseModel> page = examineMapper.paginateAs(pageNo, pageSize, queryWrapper, ExamineResponseModel.class);

        return page;
    }

    @Override
    public List<ExamineResponseModel> examineList(ExamineRequestModel request) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(EXAMINE_ENTITY.DELETED.eq(0));
        if(StrUtil.isNotEmpty(request.getName())) {
            queryWrapper.where(EXAMINE_ENTITY.NAME.like("%"+request.getName()+"%"));
        }

        return examineMapper.selectListByQueryAs(queryWrapper, ExamineResponseModel.class);
    }

    @Override
    public ExamineResponseModel examineDetail(Integer examineId) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(EXAMINE_ENTITY.ID.eq(examineId).and(EXAMINE_ENTITY.DELETED.eq(0)));
        return examineMapper.selectOneByQueryAs(queryWrapper, ExamineResponseModel.class);
    }

    @Override
    public void updateExamineStatus(Integer examineId, Integer status) {
        UpdateChain.of(ExamineEntity.class)
                .set(EXAMINE_ENTITY.STATUS, status)
                .set(EXAMINE_ENTITY.UPDATE_TIME, new Date())
                .where(EXAMINE_ENTITY.ID.eq(examineId))
                .update();
    }

    @Override
    public void deleteExamine(Integer examineId) {
        UpdateChain.of(ExamineEntity.class)
                .set(EXAMINE_ENTITY.DELETED, 1)
                .set(EXAMINE_ENTITY.UPDATE_TIME, new Date())
                .where(EXAMINE_ENTITY.ID.eq(examineId))
                .update();
    }

    @Override
    public void saveExamine(ExamineSaveModel dto) {
        ExamineEntity entity = BeanUtil.toBean(dto, ExamineEntity.class);
        entity.setUpdateTime(new Date());
        if(Objects.isNull(entity.getId())) {
            entity.setStatus(0);
            entity.setDeleted(0);
        }
        examineMapper.insertOrUpdateSelective(entity);
        dto.setId(entity.getId());
    }

    @Override
    public List<ExamineQaXrefResponseModel> examineQaXrefList(Integer examineId) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(EXAMINE_QA_XREF_ENTITY.EXAMINE_ID.eq(examineId).and(EXAMINE_QA_XREF_ENTITY.DELETED.eq(0)));
        return qaXrefMapper.selectListByQueryAs(queryWrapper, ExamineQaXrefResponseModel.class);
    }

    @Override
    public void deleteExamineQaXref(Integer examineId) {
        UpdateChain.of(ExamineQaXrefEntity.class)
                .set(EXAMINE_QA_XREF_ENTITY.DELETED, 1)
                .set(EXAMINE_QA_XREF_ENTITY.UPDATE_TIME, new Date())
                .where(EXAMINE_QA_XREF_ENTITY.EXAMINE_ID.eq(examineId).and(EXAMINE_QA_XREF_ENTITY.DELETED.eq(0)))
                .update();
    }

    @Override
    public void saveExamineQaXref(Integer examineId, List<Integer> qaIdList) {
        List<ExamineQaXrefEntity> xrefs = new ArrayList<>();
        qaIdList.forEach(qaId -> {
            ExamineQaXrefEntity xref = new ExamineQaXrefEntity();
            xref.setExamineId(examineId);
            xref.setQaId(qaId);
            xref.setDeleted(0);
            xref.setCreateTime(new Date());
            xref.setUpdateTime(new Date());
            xrefs.add(xref);
        });
        qaXrefMapper.insertBatchSelective(xrefs);
    }

    @Override
    public Page<ExamineRecordResponseModel> examineRecordPaginQuery(ExamineRecordRequestModel request) {
        Integer pageNo = ObjectUtil.defaultIfNull(request.getPageNo(), 1);
        Integer pageSize = ObjectUtil.defaultIfNull(request.getPageSize(), 10);

        QueryWrapper queryWrapper = QueryWrapper.create();
        if(StrUtil.isNotEmpty(request.getUserId())) {
            queryWrapper.where(USER_EXAMINE_RECORD_ENTITY.USER_ID.eq(request.getUserId()));
        }
        if(Objects.nonNull(request.getExamineMode())) {
            queryWrapper.where(USER_EXAMINE_RECORD_ENTITY.EXAMINE_MODE.eq(request.getExamineMode()));
        }
        if(Objects.nonNull(request.getExamineId())) {
            queryWrapper.where(USER_EXAMINE_RECORD_ENTITY.EXAMINE_ID.eq(request.getExamineId()));
        }
        if(Objects.nonNull(request.getStartTime())) {
            queryWrapper.where(USER_EXAMINE_RECORD_ENTITY.CREATE_TIME.ge(request.getStartTime()));
        }
        if(Objects.nonNull(request.getEndTime())) {
            queryWrapper.where(USER_EXAMINE_RECORD_ENTITY.CREATE_TIME.le(request.getEndTime()));
        }
        queryWrapper.where(USER_EXAMINE_RECORD_ENTITY.DELETED.eq(0));
        queryWrapper.orderBy(USER_EXAMINE_RECORD_ENTITY.CREATE_TIME, false);

        Page<ExamineRecordResponseModel> page = userExamineRecordMapper.paginateAs(pageNo, pageSize, queryWrapper, ExamineRecordResponseModel.class);

        return page;
    }

    @Override
    public Page<ExamineRecordResponseModel> examineRecordMobilePaginQuery(ExamineRecordRequestModel request) {
        Integer pageNo = ObjectUtil.defaultIfNull(request.getPageNo(), 1);
        Integer pageSize = ObjectUtil.defaultIfNull(request.getPageSize(), 10);

        QueryWrapper queryWrapper = QueryWrapper.create();
        if(StrUtil.isNotEmpty(request.getUserId())) {
            queryWrapper.where(USER_EXAMINE_RECORD_ENTITY.USER_ID.eq(request.getUserId()));
        }
        if(Objects.nonNull(request.getExamineMode())) {
            queryWrapper.where(USER_EXAMINE_RECORD_ENTITY.EXAMINE_MODE.eq(request.getExamineMode()));
        }
        if(Objects.nonNull(request.getExamineId())) {
            queryWrapper.where(USER_EXAMINE_RECORD_ENTITY.EXAMINE_ID.eq(request.getExamineId()));
        }
        queryWrapper.where(USER_EXAMINE_RECORD_ENTITY.DELETED.eq(0));
        queryWrapper.orderBy(USER_EXAMINE_RECORD_ENTITY.CREATE_TIME, false);

        Page<ExamineRecordResponseModel> page = userExamineRecordMapper.paginateAs(pageNo, pageSize, queryWrapper, ExamineRecordResponseModel.class);

        return page;
    }

    @Override
    public List<ExamineRecordResponseModel> examineRecordList(ExamineRecordRequestModel request) {

        QueryWrapper queryWrapper = QueryWrapper.create();
        if(StrUtil.isNotEmpty(request.getUserId())) {
            queryWrapper.where(USER_EXAMINE_RECORD_ENTITY.USER_ID.eq(request.getUserId()));
        }
        if(Objects.nonNull(request.getExamineMode())) {
            queryWrapper.where(USER_EXAMINE_RECORD_ENTITY.EXAMINE_MODE.eq(request.getExamineMode()));
        }
        if(Objects.nonNull(request.getExamineId())) {
            queryWrapper.where(USER_EXAMINE_RECORD_ENTITY.EXAMINE_ID.eq(request.getExamineId()));
        }
        queryWrapper.where(USER_EXAMINE_RECORD_ENTITY.DELETED.eq(0));
        return userExamineRecordMapper.selectListByQueryAs(queryWrapper, ExamineRecordResponseModel.class);
    }

    @Override
    public List<ExamineDetailRecordResponseModel> examineDetailRecordList(ExamineDetailRecordRequestModel request) {

        QueryWrapper queryWrapper = QueryWrapper.create();
        if(Objects.nonNull(request.getRecordId())) {
            queryWrapper.where(USER_EXAMINE_DETAIL_RECORD_ENTITY.RECORD_ID.eq(request.getRecordId()));
        }
        queryWrapper.orderBy(USER_EXAMINE_DETAIL_RECORD_ENTITY.CREATE_TIME, false);
        return userExamineDetailRecordMapper.selectListByQueryAs(queryWrapper, ExamineDetailRecordResponseModel.class);
    }

    @Override
    public ExamineRecordResponseModel examineRecordDetail(Integer id) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(USER_EXAMINE_RECORD_ENTITY.ID.eq(id));
        return userExamineRecordMapper.selectOneByQueryAs(queryWrapper, ExamineRecordResponseModel.class);
    }

    @Override
    public ExamineRecordResponseModel examineRecordDetail(String businessNo) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(USER_EXAMINE_RECORD_ENTITY.BUSINESS_NO.eq(businessNo));
        return userExamineRecordMapper.selectOneByQueryAs(queryWrapper, ExamineRecordResponseModel.class);
    }

    @Override
    public void saveExamineRecord(ExamineRecordSaveModel dto) {

        UserExamineRecordEntity entity = BeanUtil.toBean(dto, UserExamineRecordEntity.class);
        entity.setUpdateTime(new Date());
        userExamineRecordMapper.insertOrUpdateSelective(entity);
        dto.setId(entity.getId());
    }

    @Override
    public void saveExamineRecordDetail(ExamineRecordDetailSaveModel dto) {

        UserExamineDetailRecordEntity entity = BeanUtil.toBean(dto, UserExamineDetailRecordEntity.class);
        entity.setUpdateTime(new Date());
        userExamineDetailRecordMapper.insertOrUpdateSelective(entity);
        dto.setId(entity.getId());
    }

    @Override
    public ExamineRecordDetailResponseModel singleExamineQaDetail(Integer id) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(USER_EXAMINE_DETAIL_RECORD_ENTITY.ID.eq(id));
        return userExamineDetailRecordMapper.selectOneByQueryAs(queryWrapper, ExamineRecordDetailResponseModel.class);
    }

    @Override
    public long countUnfinishedExamineRecord(Integer examineId) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(USER_EXAMINE_DETAIL_RECORD_ENTITY.RECORD_ID.eq(examineId))
                .and(USER_EXAMINE_DETAIL_RECORD_ENTITY.EVALUATION.isNull().or(USER_EXAMINE_DETAIL_RECORD_ENTITY.SCORE.isNull()));
        return userExamineDetailRecordMapper.selectCountByQuery(queryWrapper);
    }

    @Override
    public Page<WikiCategoryResponseModel> wikiCategoryPaginQuery(WikiCategoryRequestModel request) {
        Integer pageNo = ObjectUtil.defaultIfNull(request.getPageNo(), 1);
        Integer pageSize = ObjectUtil.defaultIfNull(request.getPageSize(), 10);

        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(WIKI_CATEGORY_ENTITY.DELETED.eq(0));
        if(StrUtil.isNotEmpty(request.getName())) {
            queryWrapper.where(WIKI_CATEGORY_ENTITY.NAME.like("%"+request.getName()+"%"));
        }

        Page<WikiCategoryResponseModel> page = wikiCategoryMapper.paginateAs(pageNo, pageSize, queryWrapper, WikiCategoryResponseModel.class);

        return page;
    }

    @Override
    public List<WikiCategoryResponseModel> wikiCategoryList(WikiCategoryRequestModel request) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(WIKI_CATEGORY_ENTITY.DELETED.eq(0));
        if(StrUtil.isNotEmpty(request.getName())) {
            queryWrapper.where(WIKI_CATEGORY_ENTITY.NAME.like("%"+request.getName()+"%"));
        }

        return wikiCategoryMapper.selectListByQueryAs(queryWrapper, WikiCategoryResponseModel.class);
    }

    @Override
    public WikiCategoryResponseModel wikiCategoryDetail(Integer id) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(WIKI_CATEGORY_ENTITY.ID.eq(id).and(WIKI_CATEGORY_ENTITY.DELETED.eq(0)));
        return wikiCategoryMapper.selectOneByQueryAs(queryWrapper, WikiCategoryResponseModel.class);
    }

    @Override
    public void updateWikiCategoryStatus(Integer id, Integer status) {
        UpdateChain.of(WikiCategoryEntity.class)
                .set(WIKI_CATEGORY_ENTITY.STATUS, status)
                .set(WIKI_CATEGORY_ENTITY.UPDATE_TIME, new Date())
                .where(WIKI_CATEGORY_ENTITY.ID.eq(id))
                .update();
    }

    @Override
    public void deleteWikiCategory(Integer id) {
        UpdateChain.of(WikiCategoryEntity.class)
                .set(WIKI_CATEGORY_ENTITY.DELETED, 1)
                .set(WIKI_CATEGORY_ENTITY.UPDATE_TIME, new Date())
                .where(WIKI_CATEGORY_ENTITY.ID.eq(id))
                .update();
    }

    @Override
    public void saveWikiCategory(WikiCategorySaveModel dto) {
        WikiCategoryEntity entity = BeanUtil.toBean(dto, WikiCategoryEntity.class);
        entity.setUpdateTime(new Date());
        if(Objects.isNull(entity.getId())) {
            entity.setStatus(0);
            entity.setDeleted(0);
        }
        wikiCategoryMapper.insertOrUpdateSelective(entity);
    }

    @Override
    public Page<WikiResponseModel> wikiPaginQuery(WikiRequestModel request) {
        Integer pageNo = ObjectUtil.defaultIfNull(request.getPageNo(), 1);
        Integer pageSize = ObjectUtil.defaultIfNull(request.getPageSize(), 10);

        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(WIKI_ENTITY.DELETED.eq(0));
        if(StrUtil.isNotEmpty(request.getName())) {
            queryWrapper.where(WIKI_ENTITY.NAME.like("%"+request.getName()+"%"));
        }
        if(Objects.nonNull(request.getCategoryId())) {
            queryWrapper.where(WIKI_ENTITY.WIKI_CATEGORY_ID.eq(request.getCategoryId()));
        }

        Page<WikiResponseModel> page = wikiMapper.paginateAs(pageNo, pageSize, queryWrapper, WikiResponseModel.class);

        return page;
    }

    @Override
    public List<WikiResponseModel> wikiList(WikiRequestModel request) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(WIKI_ENTITY.DELETED.eq(0));
        if(StrUtil.isNotEmpty(request.getName())) {
            queryWrapper.where(WIKI_ENTITY.NAME.like("%"+request.getName()+"%"));
        }

        return wikiMapper.selectListByQueryAs(queryWrapper, WikiResponseModel.class);
    }

    @Override
    public WikiResponseModel wikiDetail(Integer id) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(WIKI_ENTITY.ID.eq(id).and(WIKI_ENTITY.DELETED.eq(0)));
        return wikiMapper.selectOneByQueryAs(queryWrapper, WikiResponseModel.class);
    }

    @Override
    public void updateWikiStatus(Integer id, Integer status) {
        UpdateChain.of(WikiEntity.class)
                .set(WIKI_ENTITY.STATUS, status)
                .set(WIKI_ENTITY.UPDATE_TIME, new Date())
                .where(WIKI_ENTITY.ID.eq(id))
                .update();
    }

    @Override
    public void deleteWiki(Integer id) {
        UpdateChain.of(WikiEntity.class)
                .set(WIKI_ENTITY.DELETED, 1)
                .set(WIKI_ENTITY.UPDATE_TIME, new Date())
                .where(WIKI_ENTITY.ID.eq(id))
                .update();
    }

    @Override
    public void saveWiki(WikiSaveModel dto) {
        WikiEntity entity = BeanUtil.toBean(dto, WikiEntity.class);
        entity.setUpdateTime(new Date());
        if(Objects.isNull(entity.getId())) {
            entity.setStatus(0);
            entity.setDeleted(0);
        }
        wikiMapper.insertOrUpdateSelective(entity);
    }
    
}
