package cn.breeze.elleai.job;

import cn.breeze.elleai.application.dto.inner.AiSingleEvaluateResultDto;
import cn.breeze.elleai.application.service.AiPlatformExtensionService;
import cn.breeze.elleai.application.service.AppExamineService;
import cn.breeze.elleai.domain.sparring.model.request.ExamineDetailRecordRequestModel;
import cn.breeze.elleai.domain.sparring.model.request.ExamineEvaluateJobSaveModel;
import cn.breeze.elleai.domain.sparring.model.request.ExamineRecordDetailSaveModel;
import cn.breeze.elleai.domain.sparring.model.request.ExamineRecordSaveModel;
import cn.breeze.elleai.domain.sparring.model.response.ExamineDetailRecordResponseModel;
import cn.breeze.elleai.domain.sparring.model.response.ExamineEvaluateJobResponseModel;
import cn.breeze.elleai.domain.sparring.model.response.ExamineQaResponseModel;
import cn.breeze.elleai.domain.sparring.model.response.ExamineRecordDetailResponseModel;
import cn.breeze.elleai.domain.sparring.service.CommonService;
import cn.breeze.elleai.domain.sparring.service.ExamineService;
import cn.breeze.elleai.util.Codes;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.StrUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.scheduling.annotation.Async;
import org.springframework.scheduling.quartz.QuartzJobBean;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.toMap;

@Slf4j
@Component
@RequiredArgsConstructor
public class SingleJob extends QuartzJobBean {

    private final AppExamineService appEamineService;

    private final ExamineService examineService;

    private final CommonService commonService;

    private final AiPlatformExtensionService extensionService;

    private final StringRedisTemplate redisTemplate;

    @Override
    protected void executeInternal(JobExecutionContext jobExecutionContext) throws JobExecutionException {

        log.info("定时轮询AI点评异常任务开始.");

        List<ExamineEvaluateJobResponseModel> jobList = commonService.pendingEvaluateJobList();

        log.info("定时轮询AI点评异常任务，jobList size = {}", jobList.size());
        if(CollectionUtil.isNotEmpty(jobList)) {
            jobList.forEach(v -> {
                try {
                    executeEvaluateJob(v);
                } catch (Exception ex) {
                    ex.printStackTrace();
                }
            });
        }

        log.info("定时轮询AI点评异常任务结束.");
    }

    @Async
    public void executeEvaluateJob(ExamineEvaluateJobResponseModel job) {
        Integer type = job.getType();
        Integer businessId = job.getBusinessId();
        String businessNo = job.getBusinessNo();

        // 单题点评
        if(Objects.equals(type, 0)) {
            // 获取单题答题信息
            ExamineRecordDetailResponseModel singleQaDetail = examineService.singleExamineQaDetail(businessId);

            Integer qaId = singleQaDetail.getQaId();
            ExamineQaResponseModel qaDetail = examineService.examineQaDetail(qaId);

            AiSingleEvaluateResultDto evaluateResult = extensionService.run4SingleEvaluate(businessNo, job.getUserId(), qaDetail.getQuestion(), qaDetail.getAnswer(), singleQaDetail.getAnswer());
            if(Objects.nonNull(evaluateResult)) {
                // 更新答题点评信息
                ExamineRecordDetailSaveModel detailSaveModel = new ExamineRecordDetailSaveModel();
                detailSaveModel.setId(singleQaDetail.getId());
                detailSaveModel.setScore(evaluateResult.getScore());
                detailSaveModel.setEvaluation(evaluateResult.getEvaluation());
                examineService.saveExamineRecordDetail(detailSaveModel);

                // 更新任务执行状态
                ExamineEvaluateJobSaveModel update = new ExamineEvaluateJobSaveModel();
                update.setId(job.getId());
                update.setStatus(1);
                commonService.saveEvaluateJob(update);
            }
        } else {
            // 判断所有题目是否已经单题点评完毕
            long count = examineService.countUnfinishedExamineRecord(businessId);
            if(Objects.equals(count, 0L)) {
                // 获取答题结果
                ExamineDetailRecordRequestModel requestModel = new ExamineDetailRecordRequestModel();
                requestModel.setRecordId(businessId);
                List<ExamineDetailRecordResponseModel> detailRecordList = examineService.examineDetailRecordList(requestModel);
                if(CollectionUtil.isNotEmpty(detailRecordList)) {
                    // 计算综合评分
                    double totalScore = detailRecordList.stream().filter(v -> Objects.nonNull(v.getScore())).mapToDouble(ExamineDetailRecordResponseModel::getScore).sum();
                    Double avgScore = totalScore / detailRecordList.size();

                    // 执行综合点评
                    String overallEvaluation = extensionService.run4TotalEvaluate(businessNo, job.getUserId(), businessNo);

                    if(StrUtil.isNotEmpty(overallEvaluation)) {
                        // 更新
                        ExamineRecordSaveModel saveModel = new ExamineRecordSaveModel();
                        saveModel.setId(businessId);
                        saveModel.setOverallEvaluation(overallEvaluation);
                        saveModel.setOverallScore(avgScore.floatValue());

                        examineService.saveExamineRecord(saveModel);

                        // 更新任务执行状态
                        ExamineEvaluateJobSaveModel update = new ExamineEvaluateJobSaveModel();
                        update.setId(job.getId());
                        update.setStatus(1);
                        commonService.saveEvaluateJob(update);

                        // 更新任务缓存信息
                        String totalJobStatus = String.format(Codes.TOTAL_AI_EVALUATE_JOB_STATUS, businessNo);
                        redisTemplate.opsForValue().set(totalJobStatus, "1", 7, TimeUnit.DAYS);
                    }
                }
            }
        }
    }
}
