package cn.breeze.elleai.application.service;

import cn.breeze.elleai.application.dto.PageResult;
import cn.breeze.elleai.application.dto.inner.PinYin;
import cn.breeze.elleai.application.dto.inner.Term;
import cn.breeze.elleai.application.dto.request.ProperNounRequestDto;
import cn.breeze.elleai.application.dto.request.ProperNounSaveDto;
import cn.breeze.elleai.application.dto.response.AppRoleDto;
import cn.breeze.elleai.application.dto.response.KnowledgeDataBaseCategoryDto;
import cn.breeze.elleai.application.dto.response.ProperNounDto;
import cn.breeze.elleai.domain.sparring.model.request.ProperNounRequestModel;
import cn.breeze.elleai.domain.sparring.model.request.ProperNounSaveModel;
import cn.breeze.elleai.domain.sparring.model.response.DataDicResponseModel;
import cn.breeze.elleai.domain.sparring.model.response.ProperNounResponseModel;
import cn.breeze.elleai.domain.sparring.service.CommonService;
import cn.breeze.elleai.util.ChineseCharacterUtil;
import cn.breeze.elleai.util.Codes;
import cn.breeze.elleai.util.PinYinUtils;
import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.mybatisflex.core.paginate.Page;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 应用服务
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class AppCommonService {
    
    private final CommonService commonService;
    
    /************************************************** 专有名词 **************************************************/

    /**
     * 更新专有名词
     * @param id
     * @param status
     */
    public void updateProperNounStatus(Integer id, Integer status) {
        commonService.updateProperNounStatus(id, status);
    }

    /**
     * 删除专有名词
     * @param id
     */
    public void deleteProperNoun(Integer id) {
        commonService.deleteProperNoun(id);
    }

    /**
     * 获取专有名词详情
     * @param id
     * @return
     */
    public ProperNounDto properNounDetail(Integer id) {
        ProperNounResponseModel model = commonService.properNounDetail(id);
        ProperNounDto result = BeanUtil.copyProperties(model, ProperNounDto.class);
        if(StrUtil.isNotEmpty(model.getSimilarWords())) {
            result.setSimilarWordList(JSONObject.parseArray(model.getSimilarWords(), String.class));
        }
        return result;
    }


    /**
     * 保存专有名词
     * @param dto
     */
    public void saveProperNoun(ProperNounSaveDto dto) {

        ProperNounSaveModel model = BeanUtil.copyProperties(dto, ProperNounSaveModel.class);
        model.setSimilarWords(JSON.toJSONString(dto.getSimilarWordList()));
        model.setPinyin(PinYinUtils.hansToPinYin(dto.getName()));
        commonService.saveProperNoun(model);
    }


    /**
     * 专有名词分页查询
     * @param request
     * @return
     */
    public PageResult<ProperNounDto> properNounPaginQuery(ProperNounRequestDto request) {

        ProperNounRequestModel requestModel = BeanUtil.copyProperties(request, ProperNounRequestModel.class);

        Page<ProperNounResponseModel> page = commonService.properNounPaginQuery(requestModel);

        PageResult<ProperNounDto> pageResult = PageResult.of(request.getPageNo(), request.getPageSize(), (int) page.getTotalRow(), null);

        if(CollectionUtil.isNotEmpty(page.getRecords())) {
            List<ProperNounDto> dtoList = page.getRecords().stream().map(v -> {
                ProperNounDto dto = BeanUtil.copyProperties(v, ProperNounDto.class);

                if(StrUtil.isNotEmpty(v.getSimilarWords())) {
                    dto.setSimilarWordList(JSONObject.parseArray(v.getSimilarWords(), String.class));
                }
                return dto;
            }).collect(Collectors.toList());

            pageResult.setItems(dtoList);
        }
        return pageResult;
    }


    /**
     * 专有名词列表
     * @return
     */
    public List<ProperNounDto> properNounList() {

        List<ProperNounDto> resultList = null;

        ProperNounRequestModel requestModel = new ProperNounRequestModel();
        requestModel.setStatus(1);

        List<ProperNounResponseModel> modelList = commonService.properNounList(requestModel);

        if(CollectionUtil.isNotEmpty(modelList)) {
            resultList = modelList.stream().map(v -> {
                ProperNounDto dto = BeanUtil.copyProperties(v, ProperNounDto.class);

                if(StrUtil.isNotEmpty(v.getSimilarWords())) {
                    dto.setSimilarWordList(JSONObject.parseArray(v.getSimilarWords(), String.class));
                }

                return dto;
            }).collect(Collectors.toList());
        }
        return resultList;
    }

    /**
     * 知识库分类列表
     * @return
     */
    public List<KnowledgeDataBaseCategoryDto> KnowledgeDataBaseCategoryList() {

        DataDicResponseModel dataDicByCode = commonService.getDataDicByCode(Codes.DATA_DIC_KDB_CATEGORY_LIST);
        if(Objects.nonNull(dataDicByCode) && StrUtil.isNotEmpty(dataDicByCode.getDataValue())) {
            return JSONObject.parseArray(dataDicByCode.getDataValue(), KnowledgeDataBaseCategoryDto.class);
        }
        return null;
    }

    /**
     * APP角色列表
     * @return
     */
    public List<AppRoleDto> appRoleList() {

        DataDicResponseModel dataDicByCode = commonService.getDataDicByCode(Codes.DATA_DIC_APP_ROLE_LIST);
        if(Objects.nonNull(dataDicByCode) && StrUtil.isNotEmpty(dataDicByCode.getDataValue())) {
            return JSONObject.parseArray(dataDicByCode.getDataValue(), AppRoleDto.class);
        }
        return null;
    }

    /**
     * 易错词纠正
     * @param sentence
     * @return
     */
    public String sentenceWordCorrect(String sentence) {
        PinYin pinYin = PinYinUtils.hansToPinYinObj(sentence);
        String corrected = pinYin.getSentence();
        List<ProperNounDto> filterWords = this.properNounList();
        if (CollUtil.isNotEmpty(filterWords)) {
            //先判断转化后的拼音句子是否包含了过滤词的拼音
            List<ProperNounDto> found = new ArrayList<>();
            for (ProperNounDto filterWord : filterWords) {
                if (StrUtil.contains(pinYin.getPinyin(), filterWord.getPinyin())) {
                    found.add(filterWord);
                }
            }
            if (CollUtil.isNotEmpty(found)) {
                //需要纠正易错词
                List<Term> terms = pinYin.getTerms();
                for (ProperNounDto word : found) {
                    String py = word.getPinyin();
                    List<String> samples = word.getSimilarWordList();
                    StringBuilder sb = new StringBuilder();
                    StringBuilder hans = new StringBuilder();
                    for (Term term : terms) {
                        //拼音匹配开始
                        if (StrUtil.startWith(py, term.getPinyin())) {
                            sb.append(term.getPinyin());
                            hans.append(term.getHans());
                        } else {
                            //判断是否空格，若为空格，需要支持忽略后匹配并替换
                            String current;
                            if (StrUtil.isBlank(term.getPinyin())) {
                                if (!sb.isEmpty()) {
                                    hans.append(term.getHans());
                                }
                            } else {
                                sb.append(term.getPinyin());
                                current = sb.toString();
                                if (StrUtil.equals(py, current)) {
                                    //拼接词的拼音等于易错词的拼音或者是易错词的拼音一部分
                                    hans.append(term.getHans());
                                    break;
                                    //结束匹配，把word作为
                                } else if (StrUtil.startWith(py, current)) {
                                    hans.append(term.getHans());
                                } else {
                                    //匹配失败，重置缓存
                                    sb = new StringBuilder();
                                    hans = new StringBuilder();
                                }
                            }
                        }
                    }
                    if (!sb.isEmpty()) {
                        log.info("匹配易错词[{}]，当前匹配汉字为:{}， 拼音为:{}", word.getName(), hans, sb);
                        String hansStr = hans.toString().replace(" ", "");
                        if (CollUtil.contains(samples, hansStr)) {
                            corrected = StrUtil.replaceFirst(corrected, hans.toString(), word.getName());
                        }
                    }
                }
            }
        }
        return corrected;
    }

}
