package cn.breeze.elleai.application.service;

import cn.breeze.elleai.application.dto.PageResult;
import cn.breeze.elleai.application.dto.langchain.VectorSegment;
import cn.breeze.elleai.application.dto.request.KbRequestDto;
import cn.breeze.elleai.application.dto.request.KbSaveDto;
import cn.breeze.elleai.application.dto.response.KbDto;
import cn.breeze.elleai.application.dto.response.KbTagDto;
import cn.breeze.elleai.domain.sparring.model.request.KbRequestModel;
import cn.breeze.elleai.domain.sparring.model.request.KbSaveModel;
import cn.breeze.elleai.domain.sparring.model.request.KbVectorSaveModel;
import cn.breeze.elleai.domain.sparring.model.response.KbResponseModel;
import cn.breeze.elleai.domain.sparring.model.response.KbVectorResponseModel;
import cn.breeze.elleai.domain.sparring.service.KbService;
import cn.breeze.elleai.domain.sparring.service.KbTagService;
import cn.breeze.elleai.domain.sparring.service.KbVectorService;
import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import com.mybatisflex.core.paginate.Page;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 应用服务
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class AppKbService {
    
    private final KbService kbService;

    private final KbTagService kbTagService;

    private final KbVectorService kbVectorService;

    private final AIService aiService;
    
    /************************************************** 知识库 **************************************************/

    /**
     * 更新知识库
     * @param id
     * @param status
     */
    public void updateKbStatus(Integer id, Integer status) {

        KbResponseModel kbResponseModel = kbService.kbDetail(id);
        if(Objects.nonNull(kbResponseModel)) {
            // 更新知识库状态
            kbService.updateKbStatus(id, status);

            // 启用/禁用 同步更新向量数据库
            KbVectorResponseModel vectorResponseModel = kbVectorService.kbVectorDetail(id);
            // 删除关联表
            if(Objects.nonNull(vectorResponseModel)) {
                kbVectorService.deleteKbVector(id);
                aiService.removeSegments(List.of(String.valueOf(vectorResponseModel.getVectorId())));
            }
            // 启用需要更新数据到向量数据库
            if(Objects.equals(status, 1)) {
                VectorSegment vectorSegment = new VectorSegment();
                vectorSegment.setContent(kbResponseModel.getQuestion() + "\n" + kbResponseModel.getAnswer());
                Map<String, Object> metadata = new HashMap<>();
                metadata.put("kb_id", id);
                metadata.put("tag_id", kbResponseModel.getTagId());
                metadata.put("question", kbResponseModel.getQuestion());
                metadata.put("answer", kbResponseModel.getAnswer());
                vectorSegment.setMetadata(metadata);
                List<String> strings = aiService.addVectorSegments(List.of(vectorSegment));
                // 更新segmentId到vector
                KbVectorSaveModel saveModel = new KbVectorSaveModel();
                saveModel.setVectorId(strings.get(0));
                saveModel.setKbId(id);
                kbVectorService.saveKbVector(saveModel);
            }
        }
    }

    /**
     * 删除知识库
     * @param id
     */
    public void deleteKb(Integer id) {
        kbService.deleteKb(id);
        KbVectorResponseModel vectorResponseModel = kbVectorService.kbVectorDetail(id);
        if(Objects.nonNull(vectorResponseModel)) {
            aiService.removeSegments(List.of(String.valueOf(vectorResponseModel.getVectorId())));
        }
    }

    /**
     * 获取知识库详情
     * @param id
     * @return
     */
    public KbDto kbDetail(Integer id) {
        KbResponseModel model = kbService.kbDetail(id);
        KbDto result = BeanUtil.copyProperties(model, KbDto.class);
        return result;
    }


    /**
     * 保存知识库
     * @param dto
     */
    public void saveKb(KbSaveDto dto) {
        KbSaveModel model = BeanUtil.copyProperties(dto, KbSaveModel.class);
        kbService.saveKb(model);
    }


    /**
     * 知识库分页查询
     * @param request
     * @return
     */
    public PageResult<KbDto> kbPaginQuery(KbRequestDto request) {

        KbRequestModel requestModel = BeanUtil.copyProperties(request, KbRequestModel.class);

        Page<KbResponseModel> page = kbService.kbPaginQuery(requestModel);

        PageResult<KbDto> pageResult = PageResult.of(request.getPageNo(), request.getPageSize(), (int) page.getTotalRow(), null);

        if(CollectionUtil.isNotEmpty(page.getRecords())) {
            List<KbDto> dtoList = page.getRecords().stream().map(v -> {
                KbDto dto = BeanUtil.copyProperties(v, KbDto.class);
                return dto;
            }).collect(Collectors.toList());

            pageResult.setItems(dtoList);
        }
        return pageResult;
    }


    /**
     * 初始化数据到向量数据库
     * @return
     */
    public void syncStore() {
        boolean hasNext;
        Integer pageNo = 1;
        do {
            hasNext = this.syncStoreByPage(pageNo);
            pageNo += 1;
        } while (hasNext);
    }

    @Async
    public boolean syncStoreByPage(Integer pageNo) {
        KbRequestModel requestModel = new KbRequestModel();
        requestModel.setStatus(1);
        requestModel.setPageNo(pageNo);
        requestModel.setPageSize(20);
        Page<KbResponseModel> page = kbService.kbPaginQuery(requestModel);
        if(CollectionUtil.isNotEmpty(page.getRecords())) {

            List<VectorSegment> segmentList = new ArrayList<>();

            page.getRecords().forEach(v -> {
                VectorSegment vectorSegment = new VectorSegment();
                vectorSegment.setContent(v.getQuestion() + "\n" + v.getAnswer());
                Map<String, Object> metadata = new HashMap<>();
                metadata.put("kb_id", v.getId());
                metadata.put("tag_id", v.getTagId());
                metadata.put("question", v.getQuestion());
                metadata.put("answer", v.getAnswer());
                vectorSegment.setMetadata(metadata);

                segmentList.add(vectorSegment);
            });
            // 批量更新
            List<KbVectorSaveModel> kbVectorList = new ArrayList<>();

            // 更新向量ID
            List<String> segmentIdList = aiService.addVectorSegments(segmentList);
            if(CollectionUtil.isNotEmpty(segmentIdList)) {
                for(int i=0; i<segmentIdList.size(); i++) {
                    KbResponseModel kb = page.getRecords().get(i);
                    if(Objects.nonNull(kb)) {
                        // 中间表
                        KbVectorSaveModel saveModel = new KbVectorSaveModel();
                        saveModel.setVectorId(segmentIdList.get(i));
                        saveModel.setKbId(kb.getId());
                        kbVectorList.add(saveModel);
                    }
                }
            }
            // 批量插入向量结果
            kbVectorService.batchSaveKbVector(kbVectorList);
            return true;
        }
        return false;
    }


    /**
     * 知识库分类列表
     * @return
     */
    public List<KbTagDto> kbTagList() {

//        DataDicResponseModel dataDicByCode = commonService.getDataDicByCode(Codes.DATA_DIC_KDB_CATEGORY_LIST);
//        if(Objects.nonNull(dataDicByCode) && StrUtil.isNotEmpty(dataDicByCode.getDataValue())) {
//            return JSONObject.parseArray(dataDicByCode.getDataValue(), KnowledgeDataBaseCategoryDto.class);
//        }
        return null;
    }
}
