package cn.breeze.elleai.controller.admin;


import cn.breeze.elleai.application.dto.ApiResponse;
import cn.breeze.elleai.application.dto.PageResult;
import cn.breeze.elleai.application.dto.request.ProperNounRequestDto;
import cn.breeze.elleai.application.dto.request.ProperNounSaveDto;
import cn.breeze.elleai.application.dto.response.AppRoleDto;
import cn.breeze.elleai.application.dto.response.FileUploadResult;
import cn.breeze.elleai.application.dto.response.KnowledgeDataBaseCategoryDto;
import cn.breeze.elleai.application.dto.response.ProperNounDto;
import cn.breeze.elleai.application.service.AppCommonService;
import cn.breeze.elleai.config.QueryParam;
import cn.breeze.elleai.exception.InternalException;
import cn.breeze.elleai.util.PrivateFileService;
import cn.hutool.core.exceptions.ExceptionUtil;
import cn.hutool.core.io.FileUtil;
import cn.hutool.core.util.StrUtil;
import com.google.common.collect.Lists;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.http.Cookie;
import jakarta.servlet.http.HttpServletResponse;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

@Slf4j
@RestController
@RequestMapping(value = "/admin/common")
@Tag(name = "管理端-公共服务", description = "公共服务")
@RequiredArgsConstructor
public class CommonController {

    private final AppCommonService commonService;

    private final PrivateFileService fileService;

    @Value("${sso.user.mock:true}")
    private String enableMock;

    @Value("${sso.domain:}")
    private String ssoBaseUrl;


    @Operation(summary = "专有名词详情")
    @GetMapping("/proper_noun/detail/{id}")
    public ApiResponse<ProperNounDto> detail(@Schema(description = "专有名词ID") @PathVariable("id") Integer id) {
        ProperNounDto detail = commonService.properNounDetail(id);
        return ApiResponse.ok(detail);
    }

    @Operation(summary = "启用/禁用专有名词")
    @PostMapping("/proper_noun/update_status/{id}/{status}")
    public ApiResponse<String> updateStatus(@Schema(description = "专有名词ID") @PathVariable("id") Integer id,
                                            @Schema(description = "状态（0禁用 1启用）") @PathVariable("status") Integer status) {
        commonService.updateProperNounStatus(id, status);
        return ApiResponse.ok("ok");
    }

    @Operation(summary = "删除专有名词")
    @PostMapping("/proper_noun/delete/{id}")
    public ApiResponse<String> delete(@Schema(description = "专有名词ID") @PathVariable("id") Integer id) {
        commonService.deleteProperNoun(id);
        return ApiResponse.ok("ok");
    }

    @Operation(summary = "保存专有名词")
    @PostMapping("/proper_noun/save")
    public ApiResponse<String> saveOrUpdate(@RequestBody ProperNounSaveDto dto) {
        commonService.saveProperNoun(dto);
        return ApiResponse.ok("ok");
    }

    @Operation(summary = "专有名词列表")
    @GetMapping("/proper_noun/list")
    public ApiResponse<PageResult<ProperNounDto>> list(@QueryParam ProperNounRequestDto request) {
        PageResult<ProperNounDto> pageResult = commonService.properNounPaginQuery(request);
        return ApiResponse.ok(pageResult);
    }


    @Operation(summary = "知识库分类列表")
    @GetMapping("/kdb_category/list")
    public ApiResponse<List<KnowledgeDataBaseCategoryDto>> kdbCategoryList() {
        List<KnowledgeDataBaseCategoryDto> result = commonService.KnowledgeDataBaseCategoryList();
        return ApiResponse.ok(result);
    }

    @Operation(summary = "伊丽汇APP角色列表")
    @GetMapping("/app_role/list")
    public ApiResponse<List<AppRoleDto>> appRoleList() {
        List<AppRoleDto> result = commonService.appRoleList();
        return ApiResponse.ok(result);
    }

    @Operation(summary = "上传")
    @PostMapping(value = "/upload")
    public ApiResponse<List<String>> upload(@RequestParam("files") MultipartFile[] files) {
        List<String> urls = Lists.newArrayList();
        // 判断文件数组长度
        if(files.length <= 0){
            return ApiResponse.fail(-1,"请选择文件");
        }
        for(MultipartFile file : files){
            File tmp = null;
            try {
                tmp = File.createTempFile("upload", "");
                file.transferTo(tmp);
                String key = UUID.randomUUID().toString() + "." + FileUtil.extName(file.getOriginalFilename());
                urls.add(fileService.upload(key, tmp));
            } catch (Exception e) {
                log.error("upload error,", e);
                throw new InternalException(1000, "上传失败:" + e.getMessage());
            } finally {
                if (tmp != null) {
                    tmp.delete();
                }
            }
        }
        return ApiResponse.ok(urls);
    }

    @Operation(summary = "上传带文件名返回")
    @PostMapping(value = "file/upload")
    public ApiResponse<List<FileUploadResult>> fileUpload(@RequestParam("files") MultipartFile[] files) {
        List<FileUploadResult> fileList = new ArrayList<>();
        // 判断文件数组长度
        if(files.length <= 0){
            return ApiResponse.fail(-1,"请选择文件");
        }
        for(MultipartFile file : files){
            File tmp = null;
            try {
                tmp = File.createTempFile("upload", "");
                file.transferTo(tmp);
                String key = UUID.randomUUID() + "." + FileUtil.extName(file.getOriginalFilename());
                fileList.add(new FileUploadResult().setFileName(file.getOriginalFilename()).setFilePath(fileService.upload(key, tmp)));
            } catch (Exception e) {
                log.error("upload error,", e);
                throw new InternalException(1000, "上传失败:" + e.getMessage());
            } finally {
                if (tmp != null) {
                    tmp.delete();
                }
            }
        }
        return ApiResponse.ok(fileList);
    }

    @Operation(summary = "单点登录")
    @GetMapping("/sso")
    public void userSSOLogin(@RequestParam("jwt") String token, HttpServletResponse response) {
        if (StrUtil.isNotBlank(token)) {
            log.info("单点登录:{}", token);
            try {
                Cookie jwt = new Cookie("jwt", token);
                jwt.setPath("/");
                jwt.setHttpOnly(true);
                response.addCookie(jwt);
                response.sendRedirect(ssoBaseUrl + "/elle-ai/1.0.0/index.html");
            } catch (IOException e) {
                log.error("单点登录异常{}", ExceptionUtil.getMessage(e));
            }
        }
    }


}
