package cn.breeze.elleai.controller.front;


import cn.breeze.elleai.application.dto.ApiResponse;
import cn.breeze.elleai.application.dto.PageResult;
import cn.breeze.elleai.application.dto.request.*;
import cn.breeze.elleai.application.dto.response.*;
import cn.breeze.elleai.application.service.AppExamineService;
import cn.breeze.elleai.config.QueryParam;
import cn.breeze.elleai.util.UserPrincipal;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.web.bind.annotation.*;

@RestController
@RequestMapping(value = "/front/examine")
@Tag(name = "移动端-AI对练")
@RequiredArgsConstructor
public class ExamineMobileController {

    private final AppExamineService examineService;

    @Operation(summary = "获取对练场景列表")
    @GetMapping("/scene/list")
    public ApiResponse<PageResult<ExamineSceneMobileDto>> list(@QueryParam ExamineSceneMobileRequestDto request) {
        PageResult<ExamineSceneMobileDto> pageResult = examineService.examineMobileScenePaginQuery(request);
        return ApiResponse.ok(pageResult);
    }

    @Operation(summary = "获取对练场景分类列表(需要过滤角色)")
    @GetMapping("/scene_category/list")
    public ApiResponse<PageResult<ExamineSceneCategoryMobileDto>> list(
            @Parameter(hidden = true) UserPrincipal userPrincipal,
            @QueryParam ExamineSceneCategoryRequestDto request) {
        PageResult<ExamineSceneCategoryMobileDto> pageResult = examineService.examineSceneCategoryMobilePaginQuery(request);
        return ApiResponse.ok(pageResult);
    }

    @Operation(summary = "获取对练列表(分类+名称搜索)")
    @GetMapping("/list")
    public ApiResponse<PageResult<ExamineMobileDto>> list(
            @Parameter(hidden = true) UserPrincipal userPrincipal,
            @QueryParam ExamineRequestDto request) {
        PageResult<ExamineMobileDto> pageResult = examineService.examineMobilePaginQuery(userPrincipal, request);
        return ApiResponse.ok(pageResult);
    }

    @Operation(summary = "获取对练详情(根据对练ID)")
    @GetMapping("/detail/{id}")
    public ApiResponse<ExamineMobileDto> detail(@Schema(description = "对练ID") @PathVariable("id") Integer id) {
        ExamineMobileDto detail = examineService.examineMobileDetail(id);
        return ApiResponse.ok(detail);
    }

    @Operation(summary = "获取用户对练记录列表")
    @GetMapping("/record/list")
    public ApiResponse<PageResult<ExamineRecordMobileDto>> recordList(
            @Parameter(hidden = true) UserPrincipal userPrincipal,
            @QueryParam ExamineRecordMobileRequestDto request) {
        PageResult<ExamineRecordMobileDto> pageResult = examineService.examineRecordMobilePaginQuery(userPrincipal, request);
        return ApiResponse.ok(pageResult);
    }

    @Operation(summary = "获取用户对练记录详情")
    @GetMapping("/record/detail/{id}")
    public ApiResponse<ExamineRecordMobileDto> recordDetail(@Schema(description = "对练记录ID") @PathVariable("id") Integer id) {
        ExamineRecordMobileDto detail = examineService.examineRecordMobileDetail(id);
        return ApiResponse.ok(detail);
    }

    @Operation(summary = "获取单题答题详情")
    @GetMapping("/single_qa/detail")
    public ApiResponse<ExamineDetailRecordMobileDto> singleQaDetail(@Schema(description = "答题记录ID") @RequestParam("business_id") Integer businessId) {
        ExamineDetailRecordMobileDto detail = examineService.getExamineRecordDetail(businessId);
        return ApiResponse.ok(detail);
    }

    @Operation(summary = "获取对练记录详情")
    @GetMapping("/record/detail")
    public ApiResponse<ExamineRecordMobileDto> recordDetail(@Schema(description = "对练流水号") @RequestParam("business_no") String businessNo) {
        ExamineRecordMobileDto detail = examineService.getExamineRecordDetail(businessNo);
        return ApiResponse.ok(detail);
    }

    @Operation(summary = "开始考试，并返回考试流水号&第一道题目")
    @PostMapping(value = "/start_examine")
    public ApiResponse<ExamineStartResultDto> startExamine(@Parameter(hidden = true) UserPrincipal userPrincipal,
                                                           @RequestBody StartExamineMobileRequestDto request) {

        ExamineStartResultDto result = examineService.startExamine(userPrincipal, request);
        return ApiResponse.ok(result);
    }

    @Operation(summary = "考试答题")
    @PostMapping(value = "submit_answer")
    public ApiResponse<String> submitAnswer(@Parameter(hidden = true) UserPrincipal userPrincipal,
                                            @RequestBody SubmitAnswerMobileRequestDto request) {
        examineService.submitAnswer(userPrincipal, request);
        return ApiResponse.ok("SUCCESS");
    }


    @Operation(summary = "获取下一题目")
    @GetMapping(value = "/get_next_question")
    public ApiResponse<ExamineStartResultDto> getNextQuestion(@Parameter(hidden = true) UserPrincipal userPrincipal,
                                                              @Schema(description = "对练ID") @RequestParam("examine_id") Integer examineId,
                                                              @Schema(description = "考试流水号") @RequestParam("business_no") String businessNo) {

        ExamineStartResultDto result = examineService.getNextQuestion(userPrincipal, examineId, businessNo);
        return ApiResponse.ok(result);
    }

    /**
     * 更新最终考试结果
     * @param request
     * @return
     */
    @Operation(summary = "完成考试(触发最终结果更新)")
    @PostMapping(value = "/complete_examine")
    public ApiResponse<String> updateExamResult(UserPrincipal userPrincipal, @RequestBody ExamineUpdateResultDto request) {
        examineService.completeExamine(userPrincipal, request.getExamineId(), request.getBusinessNo());
        return ApiResponse.ok("SUCCESS");
    }

    @Operation(summary = "前端轮询，获取最终考试结果")
    @GetMapping(value = "/get_examine_final_result")
    public ApiResponse<ExamineRecordMobileDto> getExamineFinalResult(@Parameter(hidden = true) UserPrincipal userPrincipal,
                                                                     @Schema(description = "对练ID") @RequestParam("examine_id") Integer examineId,
                                                                     @Schema(description = "考试流水号") @RequestParam("business_no") String businessNo) {

        ExamineRecordMobileDto result = examineService.getExamineFinalResult(userPrincipal, examineId, businessNo);
        return ApiResponse.ok(result);
    }

    /**
     * 退出考试，清除考试记录
     * @param userPrincipal
     * @param request
     * @return
     */
    @Operation(summary = "退出考试，清除考试记录")
    @PostMapping(value = "/exit_examine")
    public ApiResponse<String> exitExam(@Parameter(hidden = true) UserPrincipal userPrincipal,
                                        @RequestBody ExitExamineMobileRequestDto request) {

        examineService.exitExamine(userPrincipal, request.getExamineId(), request.getBusinessNo());

        return ApiResponse.ok("SUCCESS");
    }
}
