package cn.breeze.elleai.domain.sparring.service;

import cn.breeze.elleai.application.dto.PageResult;
import cn.breeze.elleai.application.dto.request.QaAssistantRequestDto;
import cn.breeze.elleai.application.dto.response.HotQaMobileDto;
import cn.breeze.elleai.domain.sparring.model.request.KbRequestModel;
import cn.breeze.elleai.domain.sparring.model.request.KbSaveModel;
import cn.breeze.elleai.domain.sparring.model.response.KbResponseModel;
import cn.breeze.elleai.infra.entity.KbEntity;
import cn.breeze.elleai.infra.mapper.KbMapper;
import cn.breeze.elleai.infra.mapper.KbVectorMapper;
import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.mybatisflex.core.paginate.Page;
import com.mybatisflex.core.query.QueryWrapper;
import com.mybatisflex.core.update.UpdateChain;
import com.mybatisflex.spring.service.impl.ServiceImpl;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.Objects;

import static cn.breeze.elleai.infra.entity.table.KbTableDef.KB_ENTITY;
import static cn.breeze.elleai.infra.entity.table.KbVectorTableDef.KB_VECTOR_ENTITY;

/**
 *  服务层实现。
 *
 * @author breeze
 * @since 2024-10-21
 */
@Service
@RequiredArgsConstructor
public class KbServiceImpl extends ServiceImpl<KbMapper, KbEntity> implements KbService {

    private final KbMapper kbMapper;

    private final KbVectorMapper kbVectorMapper;

    @Override
    public Page<KbResponseModel> kbPaginQuery(KbRequestModel request) {
        Integer pageNo = ObjectUtil.defaultIfNull(request.getPageNo(), 1);
        Integer pageSize = ObjectUtil.defaultIfNull(request.getPageSize(), 10);

        QueryWrapper queryWrapper = QueryWrapper.create();
        queryWrapper.where(KB_ENTITY.DELETED.eq(0));
        if(StrUtil.isNotEmpty(request.getName())) {
            queryWrapper.and(KB_ENTITY.QUESTION.like("%"+request.getName()+"%")).or(KB_ENTITY.ANSWER.like("%"+request.getName()+"%"));
        }
        if(Objects.nonNull(request.getStatus())) {
            queryWrapper.and(KB_ENTITY.STATUS.eq(request.getStatus()));
        }
        if(Objects.nonNull(request.getTagId())) {
            queryWrapper.and(KB_ENTITY.TAG_ID.eq(request.getTagId()));
        }
        queryWrapper.orderBy(KB_ENTITY.CREATE_AT, false);

        Page<KbResponseModel> page = kbMapper.paginateAs(pageNo, pageSize, queryWrapper, KbResponseModel.class);

        return page;
    }

    @Override
    public List<KbResponseModel> kbList(KbRequestModel request) {
        QueryWrapper queryWrapper = QueryWrapper.create();
        queryWrapper.where(KB_ENTITY.DELETED.eq(0));
        if(StrUtil.isNotEmpty(request.getName())) {
            queryWrapper.and(KB_ENTITY.QUESTION.like("%"+request.getName()+"%")).or(KB_ENTITY.ANSWER.like("%"+request.getName()+"%"));
        }
        if(Objects.nonNull(request.getStatus())) {
            queryWrapper.and(KB_ENTITY.STATUS.eq(request.getStatus()));
        }
        if(Objects.nonNull(request.getTagId())) {
            queryWrapper.and(KB_ENTITY.TAG_ID.eq(request.getTagId()));
        }
        queryWrapper.orderBy(KB_ENTITY.CREATE_AT, false);

        return kbMapper.selectListByQueryAs(queryWrapper, KbResponseModel.class);
    }

    @Override
    public List<KbResponseModel> kbList(List<Integer> kbIds) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(KB_ENTITY.ID.in(kbIds).and(KB_ENTITY.DELETED.eq(0)));
        return kbMapper.selectListByQueryAs(queryWrapper, KbResponseModel.class);
    }


    @Override
    public KbResponseModel kbDetail(Integer id) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .where(KB_ENTITY.ID.eq(id).and(KB_ENTITY.DELETED.eq(0)));
        return kbMapper.selectOneByQueryAs(queryWrapper, KbResponseModel.class);
    }

    @Override
    public void updateKbStatus(Integer id, Integer status) {
        UpdateChain.of(KbEntity.class)
                .set(KB_ENTITY.STATUS, status)
                .set(KB_ENTITY.UPDATE_AT, new Date())
                .where(KB_ENTITY.ID.eq(id))
                .update();
    }

    @Override
    public void deleteKb(Integer id) {
        UpdateChain.of(KbEntity.class)
                .set(KB_ENTITY.DELETED, 1)
                .set(KB_ENTITY.UPDATE_AT, new Date())
                .where(KB_ENTITY.ID.eq(id))
                .update();
    }

    @Override
    public void saveKb(KbSaveModel dto) {
        KbEntity entity = BeanUtil.toBean(dto, KbEntity.class);
        entity.setUpdateAt(new Date());
        if(Objects.isNull(entity.getId())) {
            entity.setStatus(0);
            entity.setDeleted(0);
        }
        kbMapper.insertOrUpdateSelective(entity);
    }

    @Override
    public void updateHotQuestion(List<String> vectorIds) {
        QueryWrapper qw = QueryWrapper.create()
                .select(KB_VECTOR_ENTITY.KB_ID)
                .where(KB_VECTOR_ENTITY.VECTOR_ID.in(vectorIds));
        List<Integer> kbIds = kbVectorMapper.selectObjectListByQueryAs(qw, Integer.class);
        if (CollUtil.isNotEmpty(kbIds)) {
            UpdateChain.of(KbEntity.class)
                    .set(KB_ENTITY.HINT, KB_ENTITY.HINT.add(1))
                    .where(KB_ENTITY.ID.in(kbIds))
                    .update();
        }
    }

    @Override
    public PageResult<HotQaMobileDto> hotQaList(QaAssistantRequestDto request) {
        QueryWrapper qw = QueryWrapper.create()
                .select(KB_ENTITY.QUESTION)
                .where(KB_ENTITY.STATUS.eq(1).and(KB_ENTITY.DELETED.eq(0)))
                .orderBy(KB_ENTITY.HINT.desc())
                .limit(10);
        List<String> questions = kbMapper.selectObjectListByQueryAs(qw, String.class);
        return PageResult.of(1, 10, questions.size(), CollUtil.map(questions, HotQaMobileDto::new, true));
    }
}
