package com.hdp.pi.utils.kolon;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.SocketException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.net.ftp.FTP;
import org.apache.commons.net.ftp.FTPClient;
import org.apache.commons.net.ftp.FTPReply;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;

import au.com.bytecode.opencsv.CSVReader;
import au.com.bytecode.opencsv.CSVWriter;

public class FtpUtil {

	private static final Logger logger = LoggerFactory.getLogger(FtpUtil.class);

	@Value("${kolon.property.ftp.server}")
	private String ftpServer;

	@Value("${kolon.property.ftp.port}")
	private Integer ftpPort;

	@Value("${kolon.property.ftp.userName}")
	private String ftpUserName;

	@Value("${kolon.property.ftp.password}")
	private String ftpPassword;

	/**
	 * FTP获取文件目录
	 */
	@Value("${kolon.property.ftp.getFileDir}")
	private String getFileDir;

	/**
	 * FTP获取存放历史文件目录
	 */
	@Value("${kolon.property.ftp.getFileHistoryDir}")
	private String getFileHistoryDir;

	/**
	 * 临时路径
	 */
	@Value("${kolon.property.ftp.tempFileDir}")
	private String tempFileDir;

	/**
	 * 临时存放生成的新注册用户文件路径
	 */
	@Value("${kolon.property.ftp.tempAddFileName}")
	private String tempAddFileName;

	/**
	 * 临时存放生成的更新用户文件路径
	 */
	@Value("${kolon.property.ftp.tempUpdateFileName}")
	private String tempUpdateFileName;

	/**
	 * FTP上传文件目录
	 */
	@Value("${kolon.property.ftp.putFileDir}")
	private String putFileDir;

	/**
	 * FTP上传新注册用户的文件名称
	 */
	@Value("${kolon.property.ftp.putAddFileName}")
	private String putAddFileName;

	/**
	 * FTP上传更新用户的文件名称
	 */
	@Value("${kolon.property.ftp.putUpdateFileName}")
	private String putUpdateFileName;

	/**
	 * 获取ftp链接
	 * 
	 * @return ftpClient
	 * */
	public FTPClient getFTPClient() {
		FTPClient ftpClient = null;
		try {
			ftpClient = new FTPClient();
			ftpClient.connect(ftpServer, ftpPort);
			ftpClient.enterLocalPassiveMode();
			ftpClient.login(ftpUserName, ftpPassword);
			ftpClient.setFileType(FTP.BINARY_FILE_TYPE);
			int reply = ftpClient.getReplyCode();
			if (!FTPReply.isPositiveCompletion(reply)) {
				logger.error("FTP server refused connection.");
				System.err.println("FTP server refused connection.");
				System.exit(1);
			}
		} catch (Throwable e) {
			e.printStackTrace();
			logger.error(e.toString());
		}
		return ftpClient;
	}

	/**
	 * 读取文件数据
	 */
	public List<String[]> readCSVFile(String fileName) {
		List<String[]> list = new ArrayList<String[]>();
		FTPClient ftpClient = null;
		try {
			ftpClient = this.getFTPClient();
			InputStream csv = ftpClient.retrieveFileStream(getFileDir
					+ fileName);
			if (csv == null) {
				logger.warn("kolon数据同步失败，没有找到【" + getFileDir + fileName
						+ "】数据文件");
				return null;
			}
			InputStreamReader inputStreamReader = new InputStreamReader(csv,
					"UTF-8");
			CSVReader csvReader = new CSVReader(inputStreamReader);
			list = csvReader.readAll();
			csvReader.close();
		} catch (Throwable e) {
			e.printStackTrace();
		} finally{
			if(ftpClient!=null){
				try {
					ftpClient.disconnect();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
		return list;
	}

	/**
	 * 移动文件
	 */
	public void moveFile(String fileName) {
		FTPClient ftpClient = null;
		try {
			ftpClient = this.getFTPClient();

			// 新的文件名
			String suffix = fileName.substring(fileName.lastIndexOf("."));
			String name = fileName
					.substring(0, fileName.lastIndexOf("."));
			String newFileName = name + "_" + getTime() + suffix;

			ftpClient.rename(getFileDir + fileName, getFileHistoryDir
					+ newFileName);
		} catch (Throwable e) {
			e.printStackTrace();
		}finally{
			if(ftpClient!=null){
				try {
					ftpClient.disconnect();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}

	/**
	 * 获取CSVWrite
	 * 
	 * @param type
	 * @return
	 */
	public CSVWriter getCSVWriter(Integer type) {
		createdDirIfNotExist(tempFileDir);
		CSVWriter writer = null;
		try {
			if (type == 1) {
				writer = new CSVWriter(new FileWriter(tempAddFileName), ',',
						CSVWriter.NO_QUOTE_CHARACTER);
			} else {
				writer = new CSVWriter(new FileWriter(tempUpdateFileName), ',',
						CSVWriter.NO_QUOTE_CHARACTER);
			}

		} catch (Throwable e) {
			e.printStackTrace();
		}
		return writer;
	}

	/**
	 * 上传文件到FTP服务器
	 */
	public void uploadFile() {
		try {
			SimpleDateFormat sdf = new SimpleDateFormat("YYYY-MM-dd");
			FTPClient ftpClient = this.getFTPClient();
			ftpClient.mkd(putFileDir);
			ftpClient.changeWorkingDirectory(putFileDir);
			InputStream addInputStream = new FileInputStream(new File(
					tempAddFileName));
			InputStream updateInputStream = new FileInputStream(new File(
					tempUpdateFileName));

			ftpClient.storeFile(putAddFileName.replace("YYYY-MM-dd",sdf.format(new Date())), addInputStream);
			ftpClient.storeFile(putUpdateFileName.replace("YYYY-MM-dd",sdf.format(new Date())), updateInputStream);
		} catch (Throwable e) {
			e.printStackTrace();
			logger.error(e.toString());
		}
	}

	private String getTime() {
		DateTimeFormatter formatter = DateTimeFormatter
				.ofPattern("yyyy-MM-dd_HHmmss");
		LocalDateTime dateTime = LocalDateTime.now();
		String formattedDateTime = dateTime.format(formatter);
		return formattedDateTime;
	}

	public void createdDirIfNotExist(String path) {
		File file = new File(path);
		if (!file.exists() && !file.isDirectory()) {
			file.mkdir();
		}
	}

}
