'use strict';
var mongoose = require('mongoose');
var ForumThread = mongoose.model('ForumThread');
var ForumComment = mongoose.model('ForumComment');
var ForumShare = mongoose.model('ForumShare');

var forumUserService = require('./forumUserService');
var forumCommentService = require('./forumCommentService');

var async = require('async');
var then = require('thenjs');

var listThreadFields = {
    content:1,
    type:1,
    title:1,
    pid:1,
    from:1,
    created:1,
    recommend:1,
    praise_count:1,
    comment_count:1,
    images:1,
    comments:1,
    level:1,
    top:1,
    pv_count:1,
    address:1

};

var singleThreadFields = {
    content:1,
    type:1,
    title:1,
    pid:1,
    info:1,
    from:1,
    created:1,
    recommend:1,
    praise_count:1,
    comment_count:1,
    images:1,
    level:1,
    top:1,
    pv_count:1,
    address:1

};



var subThreadFields = {
    from:1
};

/**
 * 过滤 html 标签正则
 * @type {RegExp}
 */
var htmlElReg = /<[^>]*>/ig;

/**
 * 过滤表情正则
 * @type {RegExp}
 */
var emElReg = /\【em_([0-9]*)\】/ig;

/**
 * 帖子内容处理函数
 * @param  {String} content 原始帖子内容
 * @return {String}         处理完的帖子
 */
function handleContent(content) {
    content = content || "";
    content = content
            .replace(htmlElReg, "")
            .replace(emElReg, "");

    if (content.length > 200) {
        content = content.substr(0, 200) + "……";
    }

    return content.trim();
}

//根据发帖者分页查询话题列表
exports.findThreadByPage = function(pageNo, pageSize, q, callback) {
    then(function(cont) {
        ForumThread.find(q).populate('from').count(cont);
    }).then(function(cont, count) {
        var skip = (pageNo - 1) * pageSize;
        var limit = count - skip > pageSize ? pageSize : (count - skip);
        ForumThread.find(q).skip(skip).limit(limit).sort('-quality -created').exec(function(err, docs) {

            cont(err, count, docs);
        });
    }).then(function(cont, count, threads) {        
        var rsJson = {
            result: true,
            total: count,
            datas: threads
        };
        callback(null, rsJson);

    }).fail(function(cont, err) {
        console.error(err);
        var rsJson = {
            result: false,
            err: err
        };
        callback(err, rsJson);
    });
};

//查询评论
exports.findCommentByPage = function(pageNo, pageSize, q, callback) {

    then(function(cont) {
        ForumComment.find(q).populate('from').count(cont);
    }).then(function(cont, count) {
        var skip = (pageNo - 1) * pageSize;
        var limit = count - skip > pageSize ? pageSize : (count - skip);
        ForumComment.find(q).skip(skip).limit(limit).sort('-created').populate('from').exec(function(err, docs) {

            cont(err, count, docs);
        });
    }).then(function(cont, count, threads) {
        var rsJson = {
            result: true,
            total: count,
            datas: threads
        };
        callback(null, rsJson);

    }).fail(function(cont, err) {
        console.error(err);
        var rsJson = {
            result: false,
            err: err
        };
        callback(err, rsJson);
    });
};




//创建文章
exports.createThread = function(entity, callback) {
    if (!entity.share) {
        var forum = new ForumThread(entity);

        forum.save(function(err, forum) {
            if (err) {
                console.error(err);
                callback(err, null);
            } else {
                callback(null, forum);
            }
        });
    } else {
        var forumShare = new ForumShare(entity.share);
        forumShare.ent_code = entity.ent_code;
        forumShare.save(function(err, forumShare) {
            if (err) {
                console.error(err);
                callback(err, null);
            } else {
                entity.share = forumShare._id;
                var forum = new ForumThread(entity);
                forum.save(function(err, forum) {
                    if (err) {
                        console.error(err);
                        callback(err, null);
                    } else {
                        callback(null, forum);
                    }
                });
            }
        });

    }
};

//根据ID获取文章记录，不查评论，子话题
exports.getById = function(id, callback) {
    ForumThread.findOne({
        _id: id
    }, function(err, result) {
        if (err) {
            console.error(err);
            callback(err, null);
        } else {
            callback(null, result);
        }
    });
}
//根据ID获取文章记录，不查评论，子话题
exports.getByConditions = function(conditions, callback) {
    ForumThread.findOne(conditions, function(err, result) {
        if (err) {
            console.error(err);
            callback(err, null);
        } else {
            callback(null, result);
        }
    });
}
//根据ID获取文章
exports.getThreadById = function(tid, callback) {

    ForumThread.findOne({_id: tid}, singleThreadFields).populate('from','uid mid nickName icon exp').populate('info','name icon').populate('share').exec(function(err, doc) {
        if (err) {
            console.error(err);
            callback(err, null);
        } else {
            if (doc) {
                async.parallel([
                    function(cb) {
                        if (doc.type !== 1) {
                            cb(null, null);
                            return;
                        }
                        var conditions = {
                            thread: tid,
                            level: '1'
                            // status:1
                        };
                        //获取最新10条评论                                         
                        forumCommentService.getAllComment(conditions, 1, 10, function(err, results) {
                            if (err) {
                                console.log(err);
                                cb(err, null);
                            } else {
                                cb(null, results);
                            }
                        });
                    },
                    function(cb) {
                        //获取话题子文章列表
                        if (doc.type !== 2) {
                            cb(null, null);
                            return;
                        }
                        var conditions = {
                            pid: tid,
                            type: 2,
                            status:1
                        };
                        getAllThreadByFidHelp(conditions, 1, 10, null, function(err, threads) {
                            if (err) {
                                console.error(err);
                                cb(err, null);
                            } else {
                                // console.log(threads);
                                cb(null, threads);
                            }
                        });
                    },
                    function(cb) {
                        //获取照片墙最新文章列表
                        if (doc.type !== 3) {
                            cb(null, null);
                            return;
                        }
                        var conditions = {
                            pid: tid,
                            type: 3,
                            status:1,
                            images: {$exists: true, $not: {$size: 0}} 
                        };
                        getAllThreadByFidHelp(conditions, 1, 10, '-created', function(err, threads) {
                            if (err) {
                                console.error(err);
                                cb(err, null);
                            } else {
                                cb(null, threads);
                            }
                        });
                    },
                    function(cb) {
                        //获取照片墙最热文章列表
                        if (doc.type !== 3) {
                            cb(null, null);
                            return;
                        }
                        var conditions = {
                            pid: tid,
                            type: 3,
                            status:1,
                            images: {$exists: true, $not: {$size: 0}} 
                        };
                        getAllThreadByFidHelp(conditions, 1, 10, '-praise_count', function(err, threads) {
                            if (err) {
                                console.error(err);
                                cb(err, null);
                            } else {
                                cb(null, threads);
                            }
                        });
                    },
                    function(cb) {
                        //获取发帖人数
                        ForumThread.aggregate({
                            $match: {
                                pid: mongoose.Types.ObjectId(tid)
                            }
                        }, {
                            $group: {
                                _id: {
                                    from: '$from'
                                },
                                count: {
                                    $sum: 1
                                }
                            }
                        }, function(err, data) {
                            if (err) {
                                cb(err, null);
                            } else {
                                cb(null, data.length);
                            }
                        });
                    }
                ], function(err, results) {
                    if (err) {
                        callback(err, null);
                    } else {
                        var thread = doc;
                        var comments = results[0] || [];
                        var subThreads = results[1] || []; //话题子文章列表
                        var latestPhotos = results[2] || []; //最新照片墙列表
                        var hotPhotos = results[3] || []; //最热照片墙列表
                        var canyuPeopleCount = results[4] || 0; //参与人数

                        var threadObj = thread && thread.toObject() || {};
                        threadObj.comments = comments;
                        threadObj.subThreads = subThreads;
                        threadObj.latestPhotos = latestPhotos;
                        threadObj.hotPhotos = hotPhotos;
                        threadObj.canyuPeopleCount = canyuPeopleCount;

                        callback(null, threadObj);
                    }
                });

            } else {
                callback(null, {});
            }
        }
    });
};
//根据ID获取文章
exports.getThreadByIdSimple = function(conditions, callback) {
    ForumThread.findOne(conditions).populate('from info share pid').exec(function(err, doc) {
        callback(err, doc);
    });
};
//根据ID获取文章
exports.getThreadByIdNoLimit = function(tid, callback) {
    async.parallel([
        function(cb) {
            var conditions = {
                _id: tid
            };
            ForumThread.findOne(conditions,singleThreadFields).populate('from').populate('info').populate('share').exec(function(err, doc) {
                if (err) {
                    console.error(err);
                    callback(err, null);
                } else {
                    cb(null, doc);
                }
            });
        },
        function(cb) {
            var conditions = {
                thread: tid,
                level: '1'
                // status:1
            };
            //获取最新10条评论                                         
            forumCommentService.getAllComment(conditions, 1, 999, function(err, results) {
                if (err) {
                    console.log(err);
                    cb(err, null);
                } else {
                    cb(null, results);
                }
            });
        },
        function(cb) {
            //获取话题子文章列表
            var conditions = {
                pid: tid,
                type: 2,
                status:1
            };
            getAllThreadByFidHelp(conditions, 1, 10, null, function(err, threads) {
                if (err) {
                    console.error(err);
                    cb(err, null);
                } else {
                    // console.log(threads);
                    cb(null, threads);
                }
            });
        },
        function(cb) {
            //获取照片墙最新文章列表
            var conditions = {
                pid: tid,
                type: 3,
                status:1,
                images: {$exists: true, $not: {$size: 0}} 
            };
            getAllThreadByFidHelp(conditions, 1, 10, '-created', function(err, threads) {
                if (err) {
                    console.error(err);
                    cb(err, null);
                } else {
                    cb(null, threads);
                }
            });
        },
        function(cb) {
            //获取照片墙最热文章列表
            var conditions = {
                pid: tid,
                type: 3,
                status:1,
                images: {$exists: true, $not: {$size: 0}} 
            };
            getAllThreadByFidHelp(conditions, 1, 10, '-praise_count', function(err, threads) {
                if (err) {
                    console.error(err);
                    cb(err, null);
                } else {
                    cb(null, threads);
                }
            });
        },
        function(cb) {
            //获取发帖人数
            ForumThread.aggregate({
                $match: {
                    pid: mongoose.Types.ObjectId(tid)
                }
            }, {
                $group: {
                    _id: {
                        from: '$from'
                    },
                    count: {
                        $sum: 1
                    }
                }
            }, function(err, data) {
                if (err) {
                    cb(err, null);
                } else {
                    cb(null, data.length);
                }
            });
        }
    ], function(err, results) {
        if (err) {
            callback(err, null);
        } else {
            var thread = results[0];
            var comments = results[1];
            var subThreads = results[2] || []; //话题子文章列表
            var latestPhotos = results[3] || []; //最新照片墙列表
            var hotPhotos = results[4] || []; //最热照片墙列表
            var canyuPeopleCount = results[5] || 0; //参与人数

            var threadObj = thread.toObject();
            threadObj.comments = comments;
            threadObj.subThreads = subThreads;
            threadObj.latestPhotos = latestPhotos;
            threadObj.hotPhotos = hotPhotos;
            threadObj.canyuPeopleCount = canyuPeopleCount;

            callback(null, threadObj);
        }
    });
};

//根据ID更新文章
exports.updateThreadById = function(tid, entity, callback) {
    var shareEntity = entity.share;
    if (entity.share && entity.share._id) {
        entity.share = entity.share._id;
    } else {
        entity.share = '';
    }
    if (entity.share) {

        ForumThread.update({
            _id: tid
        }, entity, null, function(err, result) {
            if (err) {
                console.error(err);
                callback(err, null);
            } else {
                ForumShare.update({
                    _id: shareEntity._id
                }, shareEntity, null, function(err, se) {
                    if (err) {
                        console.error(err);
                        callback(err, null);
                    } else {
                        callback(null, entity);
                    }
                });

            }
        });
    } else {

        if (shareEntity && (shareEntity.title || shareEntity.description || shareEntity.icon)) {
            shareEntity.ent_code = entity.ent_code;

            var forumShare = new ForumShare(shareEntity);
            forumShare.save(function(err, forumShare) {
                entity.share = forumShare._id;
                ForumThread.update({
                    _id: tid
                }, entity, null, function(err, result) {
                    if (err) {
                        console.error(err);
                        callback(err, null);
                    } else {
                        callback(null, entity);

                    }
                });
            });
        } else {
            delete entity.share;
            ForumThread.update({
                _id: tid
            }, entity, null, function(err, result) {
                if (err) {
                    console.error(err);
                    callback(err, null);
                } else {
                    callback(null, entity);

                }
            });
        }
    }
};

//逻辑删除文章
exports.logicDeleteThreadById = function(tid, callback) {
    ForumThread.update({
        _id: mongoose.Types.ObjectId(tid)
    }, {
        status : 3
    }, null, function(err, result) {
        if (err) {
            console.error(err);
            callback(err, false);
        } else {
            callback(null, true);
        }
    });
};


//根据ID更新文章
exports.deleteThreadById = function(tid, callback) {
    ForumThread.remove({
        _id: tid
    }, function(err, result) {
        if (err) {
            console.error(err);
            callback(err, null);
        } else {
            callback(null, null);
        }
    });
};

//一级评论
function populateComment(doc, callback) {
    if (doc && doc.comments.length > 0) {
        var asyncTasks = [];
        doc.comments.forEach(function(comment) {
            asyncTasks.push(function(callback) {
                ForumComment.populate(comment, {
                    path: 'from to',
                    select: 'uid  nickName icon displayName displayIcon comments'
                }, function(err, c) {
                    if (err) {
                        console.error(err);
                        callback(null, null);
                    } else {
                        // console.log(comment);
                        if (comment.comments) {
                            populateSubComment(comment, function(err, results) {
                                if (err) {
                                    console.error(err);
                                    callback(null, null);
                                } else {
                                    var newobj = comment.toObject();
                                    newobj.comments = results;
                                    callback(null, newobj);
                                }
                            });
                        } else {
                            callback(null, doc);
                        }
                    }
                });
            });
        });
        async.parallel(asyncTasks, function(err, results) {
            if (err) {
                console.log(err);
                callback(null, null);
            } else {
                if (results && results.length > 0) {
                    callback(null, results[0]);
                } else {
                    callback(null, {});
                }
                // callback(null, results);
            }
        });
    } else {
        callback(null, doc);
    }
}

//二级评论
function populateSubComment(subComments, callback) {
    if (subComments.comments && subComments.comments.length > 0) {
        var asyncTasks = [];
        subComments.comments.forEach(function(comment) {
            asyncTasks.push(function(callback) {
                ForumComment.findOne({
                    _id: comment
                }).populate({
                    path: 'from to',
                    select: 'uid nickName icon'
                }).exec(function(err, c) {
                    if (err) {
                        console.error(err);
                        callback(null, null);
                    } else {
                        callback(null, c);
                    }
                });
            });
        });
        async.parallel(asyncTasks, function(err, results) {
            if (err) {
                console.log(err);
                callback(null, null);
            } else {
                callback(null, results);
            }
        });

    } else {
        callback(null, null);
    }
}

//获取数量
function countAllByFid(conditions, callback) {
    // callback(null, 21);
    ForumThread.count(conditions, function(err, count) {
        if (err) {
            console.error(err);
            callback(err, null);
        } else {
            callback(null, count);
        }
    });
}

//导出获取数量函数
exports.getAllCountByFid = function(conditions, callback) {
    countAllByFid(conditions, callback);
}

//获取话题、照片墙子文章数据
function getSubThreads(doc, sort, callback) {
    var conditions = {
        pid: doc._id
    };
    countAllByFid(conditions, function(err, count) {
        if (err) {
            console.error(err);
            callback(err, null);
        } else {
            var sortBy = '-top -quality -created';
            if (sort) {
                sortBy = sort;
            }
            ForumThread.find(conditions, subThreadFields).populate('from', 'icon').sort(sortBy).exec(function(err, docs) {
                if (err) {
                    console.error(err);
                    callback(err, null);
                } else {
                    var obj = {};
                    obj.total = count;
                    obj.items = docs;

                    var newDoc = doc.toObject();
                    newDoc.subThreads = obj;

                    callback(null, newDoc);
                }
            });
        }
    });
}

function getAllThreadByFidHelp(conditions, pageNo, pageSize, sort, callback) {
    console.time('count_cost');
    
    countAllByFid(conditions, function(err, count) {
        if (err) {
            console.error(err);
            callback(err, null);
        } else {
            console.timeEnd('count_cost');
            console.time('threads_cost');
            var skip = (pageNo - 1) * pageSize;
            var limit = pageSize;
            var sortBy = '-topTime -tag_topTime -quality -created';
            if (sort) {
                sortBy = sort;
            }
            ForumThread.find(conditions,listThreadFields).populate('from', 'uid mid nickName icon').populate({
                path: 'comments',
                options: {
                    where:{status:1},
                    limit: 3,
                    sort: '-created'
                },
                select: 'from to created content level'
            }).limit(limit).skip(skip).sort(sortBy).exec(function(err, docs) {
                if (err) {
                    console.error(err);
                    callback(err, null);
                } else {
                    console.timeEnd('threads_cost');
                    var obj = {};
                    obj.total = count;
                    obj.pageNo = pageNo;
                    obj.pageSize = pageSize;
                    obj.items = docs;
                    console.time('comments_cost');
                    if (docs && docs.length > 0) {
                        var asyncTasks = [];

                        docs.forEach(function(doc) {
                            // console.log(doc);
                            if (doc.type === 1 || doc.level !== 1) { //非照片墙或文章时获取评论
                                asyncTasks.push(function(callback) {
                                    populateComment(doc, callback);
                                });
                            } else {
                                asyncTasks.push(function(callback) {
                                    getSubThreads(doc, null, callback);
                                });
                            }

                        });
                        async.parallel(asyncTasks, function(err, results) {
                            if (err) {
                                console.error(err);
                                callback(null, null);
                            } else {
                                obj.items = results;
                                callback(null, obj);
                                console.timeEnd('comments_cost');
                            }
                        });

                    } else {
                        callback(null, obj);
                    }
                }
            });
        }
    });
}

function getAllThreadByFidHelpNoLimit(conditions, pageNo, pageSize, sort, callback) {
    countAllByFid(conditions, function(err, count) {
        if (err) {
            console.error(err);
            callback(err, null);
        } else {
            var skip = (pageNo - 1) * pageSize;
            var limit = count - skip > pageSize ? pageSize : (count - skip);
            var sortBy = '-topTime -tag_topTime -quality -created';
            if (sort) {
                sortBy = sort;
            }
            ForumThread.find(conditions,listThreadFields).populate('from','uid mid nickName icon').populate('info','name icon pv_count').populate({
                path: 'tag',select: 'title pv_count'
            }).populate({
                path: 'comments',
                options: {
                    where:{status:1},
                    sort: 'created'
                },
                select: 'from to created content'
            }).limit(limit).skip(skip).sort(sortBy).exec(function(err, docs) {
                if (err) {
                    console.error(err);
                    callback(err, null);
                } else {
                    var obj = {};
                    obj.total = count;
                    obj.pageNo = pageNo;
                    obj.pageSize = pageSize;
                    obj.items = docs;

                    if (docs && docs.length > 0) {
                        var asyncTasks = [];

                        docs.forEach(function(doc) {
                            // console.log(doc);
                            if (doc.type === 1 || doc.level !== 1) { //非照片墙或文章时获取评论
                                asyncTasks.push(function(callback) {
                                    populateComment(doc, callback);
                                });
                            } else {
                                asyncTasks.push(function(callback) {
                                    getSubThreads(doc, null, callback);
                                });
                            }

                        });
                        async.parallel(asyncTasks, function(err, results) {
                            if (err) {
                                console.error(err);
                                callback(null, null);
                            } else {
                                obj.items = results;
                                callback(null, obj);
                            }
                        });

                    } else {
                        callback(null, obj);
                    }
                }
            });
        }
    });
}


//获取全部列表数据
exports.getAllThreadByFidNoLimit = function(conditions, pageNo, pageSize, sort, callback) {
    getAllThreadByFidHelpNoLimit(conditions, pageNo, pageSize, sort, callback);
};

//获取全部列表数据
exports.getAllThreadByFid = function(conditions, pageNo, pageSize, sort, callback) {
    redis.get('query_1000000', function(err, page){
        if (err || !page) {
            getAllThreadByFidHelp(conditions, pageNo, pageSize, sort, function(err, obj){
                redis.set('query_1000000', JSON.stringify(obj));
                callback(err, obj);
            });
        } else {
            callback(null, JSON.parse(page));
        }
    });
    
};

//根据板块ID更新板块下的top为0,并把当前文章的top设为1（置顶）
exports.updateTopByThreadId = function(infoId, threadId, callback) {
    // ForumThread.update({info:infoId}, {top:0},{multi:true}, function(err, doc) {
    //  if(err){
    //      callback(err,null);
    //  }else{
    //      ForumThread.findOneAndUpdate({_id:threadId}, {top:1}, function(err, doc) {
    //          if(err){
    //              callback(err,null);
    //          }else{
    //              callback(null,null);
    //          }
    //      });
    //  }
    // });
    ForumThread.findOneAndUpdate({
        _id: threadId
    }, {
        top: 1,
        topTime: new Date()
    }, function(err, doc) {
        if (err) {
            callback(err, null);
        } else {
            callback(null, null);
        }
    });
};

//根据板块ID更新板块下的top为0,并把当前文章的top设为1（置顶）
exports.updateUnTopByThreadId = function(infoId, threadId, callback) {
    ForumThread.findOneAndUpdate({
        _id: threadId
    }, {
        top: 0,
        topTime: null
    }, function(err, doc) {
        if (err) {
            callback(err, null);
        } else {
            callback(null, null);
        }
    });
};
//根据文章ID更新文章为推荐
exports.updateRecommendByThreadId = function(threadId, callback) {
    ForumThread.findOneAndUpdate({
        _id: threadId
    }, {
        recommend: 1
    }, function(err, doc) {
        callback(err,doc);
    });
};
//根据文章ID更新文章为不推荐
exports.updateUnRecommendByThreadId = function(threadId, callback) {
    ForumThread.findOneAndUpdate({
        _id: threadId
    }, {
        recommend: 0
    }, function(err, doc) {
        callback(err,null);
    });
};
//根据板块ID更新标签置顶（置顶）
exports.updateTagTop = function(tid,tag_top,callback){
    var time = '';
    if(tag_top == 1){
        time = new Date();
    }
    ForumThread.findOneAndUpdate({
        _id: tid
    }, {
        tag_top: tag_top,
        tag_topTime: time
    }, function(err, doc) {
        if (err) {
            callback(err, null);
        } else {
            callback(null, doc);
        }
    });
};

//更新文章点赞数
exports.updateThreadRaiseCount = function(threadId, callback) {
    ForumThread.update({
            _id: threadId
        }, {
            $inc: {
                praise_count: 1
            }
        }, {
            w: 1,
            safe: true
        },
        function(err, result) {
            if (err) {
                console.error(err);
                callback(err, null);
            } else {
                callback(null, null);
            }
        });
};

//减少文章点赞数
exports.updateThreadRaiseCountDec = function(threadId, callback) {
    ForumThread.update({
        _id: threadId
    }, {
        $inc: {
            praise_count: -1
        }
    }, {
        w: 1,
        safe: true
    },
    function(err, result) {
        if (err) {
            console.error(err);
            callback(err, null);
        } else {
            callback(null, null);
        }
    });
};

//更新文章分享数
exports.updateThreadShareCount = function(threadId, callback) {
    ForumThread.update({
            _id: threadId
        }, {
            $inc: {
                share_count: 1
            }
        }, {
            w: 1,
            safe: true
        },
        function(err, result) {
            if (err) {
                console.error(err);
                callback(err, null);
            } else {
                callback(null, null);
            }
        });
};

//更新文章评论数
exports.updateThreadCommentCountInc = function(threadId, callback) {
    ForumThread.update({
            _id: threadId
        }, {
            $inc: {
                comment_count: 1
            }
        }, {
            w: 1,
            safe: true
        },
        function(err, result) {
            if (err) {
                console.error(err);
                callback(err, null);
            } else {
                callback(null, null);
            }
        });
};

//更新文章评论数 
exports.updateThreadCommentCount = function(threadId, callback) {
    ForumThread.update({
            _id: threadId
        }, {
            $inc: {
                comment_count: -1
            }
        }, {
            w: 1,
            safe: true
        },
        function(err, result) {
            if (err) {
                console.error(err);
                callback(err, null);
            } else {
                callback(null, null);
            }
        });
};

//更新文章浏览数
exports.updateThreadPvCount = function(threadId, callback) {
    ForumThread.update({
            _id: threadId
        }, {
            $inc: {
                pv_count: 1
            }
        }, {
            w: 1,
            safe: true
        },
        function(err, result) {
            if (err) {
                console.error(err);
                callback(err, null);
            } else {
                callback(null, null);
            }
        });
};

// 查找数组元素下标
function indexOf(array,val) {
    for (var i = 0; i < array.length; i++) {
        if (array[i] == val) return i;
    }
    return -1;
};

// 删除数组指定值
exports.remove = function(array,val) {
    var index = indexOf(array,val);
    if (index > -1) {
        array.splice(index, 1);
    }
    return array;
};


function getAllThreadByFidAndUserHelp(user_ids, conditions, pageNo, pageSize, sort, callback) {
    conditions.from = {
        "$in" : user_ids
    };

    countAllByFid(conditions, function(err, count) {
        if (err) {
            console.error(err);
            callback(err, null);
        } else {

            var skip = (pageNo - 1) * pageSize;
            var limit = count - skip > pageSize ? pageSize : (count - skip);
            var sortBy = '-topTime -tag_topTime -quality -created';
            if (sort) {
                sortBy = sort;
            }
            ForumThread.find(conditions, listThreadFields).populate('from','uid mid nickName icon').populate('info','name icon pv_count').populate({
                path: 'tag', select: 'title pv_count'
            }).populate({
                path: 'comments',
                options: {
                    where:{status:1},
                    limit: 5,
                    sort: '-created'
                },
                select: 'from to created content'
            }).limit(limit).skip(skip).sort(sortBy).exec(function(err, docs) {
                if (err) {
                    console.error(err);
                    callback(err, null);
                } else {
                    var obj = {};
                    obj.total = count;
                    obj.pageNo = pageNo;
                    obj.pageSize = pageSize;
                    obj.items = docs;
                    callback(null, obj);
                }
            });
        }
    });
}

//获取全部列表数据
exports.getAllThreadByFidAndNickName = function(nickName, conditions, pageNo, pageSize, sort, callback) {
    //查询到用户
    forumUserService.searchMembersByNickName(nickName, function(err, users) {
        if (err) {
            console.error(err);
            callback(err, null);
        } else {
            //查询对应用户的文章
            var user_ids = [];
            for(var i in users){
                user_ids.push(users[i]._id);
            }
            getAllThreadByFidAndUserHelp(user_ids, conditions, pageNo, pageSize, sort, callback);
        }
    
    });
}
//获取全部列表数据
exports.getAllThreadByFidAndMid = function(mid, conditions, pageNo, pageSize, sort, callback) {
    //查询到用户
    forumUserService.searchMembersByMid(mid, function(err, users) {
        if (err) {
            console.error(err);
            callback(err, null);
        } else {
            //查询对应用户的文章
            var user_ids = [];
            for(var i in users){
                user_ids.push(users[i]._id);
            }
            getAllThreadByFidAndUserHelp(user_ids, conditions, pageNo, pageSize, sort, callback);
        }
    
    });
}


//更新文章
exports.updateThread = function(condition, entity, callback) {
    ForumThread.findOneAndUpdate(condition, entity, function(err, doc){
        if (err) {
            console.error(err);
            callback(err, null);
        } else {
            callback(null, doc);
        }
    });
};