'use strict';
var mongoose = require('mongoose');
var ForumUser = mongoose.model('ForumUser');
var then = require('thenjs');
var async = require('async');

var forumLimitOperationService = require('../service/forumLimitOperationService');

//创建用户
exports.createUser = function(entity, callback) {
    var forum = new ForumUser(entity);
    forum.save(function(err, forum) {
        if (err) {
            console.error(err);
            callback(err, null);
        } else {
            callback(null, forum);
        }
    });
};

//根据Uid获取用户
exports.getByConditionsSelectyField = function(conditions, fields, callback) {
    ForumUser.findOne(conditions).select(fields).exec(function(err, result) {
        if (err) {
            callback(err, null);
        } else {
            if (result) {
                callback(null, result);
            } else {
                callback(null, null);
            }
        }
    });
};

//根据Uid获取用户
exports.getUserByUid = function(uid, callback) {
    ForumUser.findOne({
        uid: uid
    }).populate({
        path: 'honorTitles',
        match: {
            status: {
                $ne: 0
            }
        },
        options: {
            sort: {
                order_idx: -1
            }
        }
    }).exec(function(err, result) {
        if (err) {
            callback(err, null);
        } else {
            if (result) {
                callback(null, result);
            } else {
                callback(null, null);
            }
        }
    });
};

//根据id获取用户
exports.getUserById = function(id, callback) {
    ForumUser.findOne({
        _id: id
    }).populate({
        path: 'honorTitles',
        match: {
            status: {
                $ne: 0
            }
        },
        options: {
            sort: {
                order_idx: -1
            }
        }
    }).exec(function(err, result) {
        if (err) {
            callback(err, null);
        } else {
            if (result) {
                callback(null, result);
            } else {
                callback(null, null);
            }
        }
    });
};

//根据ID更新用户信息
exports.updateUserById = function(uid, entity, callback) {
    ForumUser.update({
        _id: uid
    }, entity, null, function(err, result) {
        if (err) {
            console.error(err);
            callback(err, null);
        } else {
            callback(null, null);
        }
    });
};

//根据ID更新用户信息2
exports.updateByIdWithOptions = function(condition, entity, options, callback) {
    ForumUser.findOneAndUpdate(condition, entity, options, function(err, result) {
        if (err) {
            console.error(err);
            callback(err, null);
        } else {
            callback(null, result._id);
        }
    });
};

//查询用户信息
exports.searchMembers = function(pageNo, pageSize, q, callback) {
    then(function(cont) {
        ForumUser.find(q).count(cont);
    }).then(function(cont, count) {
        var skip = (pageNo - 1) * pageSize;
        var limit = count - skip > pageSize ? pageSize : (count - skip);
        ForumUser.find(q).skip(skip).limit(limit).sort('-created').populate({
            path: 'honorTitles',
            match: {
                status: {
                    $ne: 0
                }
            },
            options: {
                sort: {
                    order_idx: -1
                }
            }
        }).exec(function(err, docs) {
            cont(err, count, docs);
        });
    }).then(function(cont, count, members) {
        var rsJson = {
            result: true,
            total: count,
            datas: members
        };
        callback(null, rsJson);
    }).fail(function(cont, err) {
        console.error(err);
        var rsJson = {
            result: false,
            err: err
        };
        callback(err, rsJson);
    });
};

//根据nickName查询用户
exports.searchMembersByNickName = function(nickName, callback) {
    var name = {
        $or: [{
            nickName: {
                $regex: nickName,
                $options: 'i'
            }
        }, {
            displayName: {
                $regex: nickName,
                $options: 'i'
            }
        }]
    };
    ForumUser.find(name).populate({
        path: 'honorTitles',
        match: {
            status: {
                $ne: 0
            }
        },
        options: {
            sort: {
                order_idx: -1
            }
        }
    }).exec(function(err, result) {
        if (err) {
            callback(err, null);
        } else {
            callback(null, result);
        }
    });
};

//根据mid查询用户
exports.searchMembersByMid = function(mid, callback) {
    ForumUser.find({
        mid: mid
    }).populate({
        path: 'honorTitles',
        match: {
            status: {
                $ne: 0
            }
        },
        options: {
            sort: {
                order_idx: -1
            }
        }
    }).exec(function(err, result) {
        if (err) {
            callback(err, null);
        } else {
            callback(null, result);
        }
    });
};

//查询用户和行为限制信息
exports.searchMembersAndLimitActions = function(pageNo, pageSize, q, callback) {
    then(function(cont) {
        ForumUser.find(q).count(cont);
    }).then(function(cont, count) {
        var skip = (pageNo - 1) * pageSize;
        var limit = count - skip > pageSize ? pageSize : (count - skip);
        ForumUser.find(q).skip(skip).limit(limit).sort('-created').populate({
            path: 'honorTitles',
            match: {
                status: {
                    $ne: 0
                }
            },
            options: {
                sort: {
                    order_idx: -1
                }
            }
        }).exec(function(err, docs) {
            cont(err, count, docs);
        });
    }).then(function(cont, count, members) {
        if (count > 0) {
            var asyncTasks = [];
            members.forEach(function(member) {
                asyncTasks.push(function(callback) {
                    forumLimitOperationService.getLimitOperationByUserId(member._id, function(err, forumLimitOperation) {
                        if (err) {
                            console.error(err);
                            callback(null, null);
                        } else {
                            if (forumLimitOperation) {
                                var limit_actions = forumLimitOperation.limit_actions;
                                var array = [];
                                for (var i = 0; i < limit_actions.length; i += 1) {
                                    var limit_action = limit_actions[i];
                                    var name = '';
                                    switch (limit_action.limit_type) {
                                        case 1:
                                            name = '禁言';
                                            break;
                                        case 2:
                                            name = '不加积分';
                                            break;
                                    }
                                    array.push(name);
                                }
                                member = member.toObject();
                                member.limitActions = array;
                                callback(null, member);
                            } else {
                                callback(null, member);
                            }
                        }
                    });
                });
            });
            async.parallel(asyncTasks, function(err, results) {
                if (err) {
                    console.log(err);
                    callback(null, null);
                } else {
                    var rsJson = {
                        result: true,
                        total: count,
                        datas: results
                    };
                    callback(null, rsJson);
                }
            });

        } else {
            var rsJson = {
                result: true,
                total: count,
                datas: members
            };
            callback(null, rsJson);
        }
    }).fail(function(cont, err) {
        console.error(err);
        var rsJson = {
            result: false,
            err: err
        };
        callback(err, rsJson);
    });
};


//根据UID和entCode更新用户信息
exports.updateUserByUIdAndEntCode = function(condition, entity, callback) {
    ForumUser.update(condition, entity, null, function(err, result) {
        if (err) {
            console.error(err);
            callback(err, null);
        } else {
            callback(null, null);
        }
    });
};

//查询用户收藏记录
exports.getFavors = function(condition, callback) {
    ForumUser.findOne(condition).populate({
        path: 'favor_by',
        select: '_id nickName displayName'
    }).populate({
        path: 'favor_users',
        select: '_id nickName displayName'
    }).populate({
        path: 'favor_infos',
        select: '_id name',
        options: {sort: {created: -1}}
    }).populate({
        path: 'favor_threads',
        select: '_id title info from created status type',
        options: {
            sort: {created: -1},
            populate:"info from" //继续查文章里面的板块信息
        }
    }).exec(function(err, result) {
        if (err) {
            callback(err, null);
        } else {
            callback(null, result);
        }
    });
};


//查询用户信息,只查某些字段
exports.searchMembersAndFields = function(pageNo, pageSize, q, fields, callback) {
    then(function(cont) {
        ForumUser.find(q).count(cont);
    }).then(function(cont, count) {
        var skip = (pageNo - 1) * pageSize;
        var limit = count - skip > pageSize ? pageSize : (count - skip);
        ForumUser.find(q).skip(skip).limit(limit).select(fields).sort('-created').populate({
            path: 'honorTitles',
            match: {
                status: {
                    $ne: 0
                }
            },
            options: {
                sort: {
                    order_idx: -1
                }
            }
        }).exec(function(err, docs) {
            cont(err, count, docs);
        });
    }).then(function(cont, count, members) {
        var rsJson = {
            result: true,
            total: count,
            datas: members
        };
        callback(null, rsJson);
    }).fail(function(cont, err) {
        console.error(err);
        var rsJson = {
            result: false,
            err: err
        };
        callback(err, rsJson);
    });
};


//查询用户信息,只查某些字段
exports.searchMembersWithFields = function(q, fields, sort, callback) {
    ForumUser.find(q).select(fields).sort(sort).exec(function(err, docs) {
        callback(err, docs);
    });
};

//查询用户收藏记录
exports.getUserFavorThreadIds = function(q, fields, callback) {
    ForumUser.findOne(q).select(fields).exec(function(err, result) {
        if (err) {
            callback(err, null);
        } else {
            callback(null, result);
        }
    });
};