'use strict';
//插件
var express = require('express'),
    router = express.Router(),
    returnCode = require('../../utils/returnCode'),
    _ = require('lodash'),
    then = require('thenjs'),
    moment = require('moment'),
    nodeExcel = require('excel-export'),
    mongoose = require('mongoose'),
    async = require('async');
//模型    
var ForumThread = mongoose.model('ForumThread'),
    ForumPVLog = mongoose.model('ForumPVLog');
//服务    
var forumThreadService = require('../../service/forumThreadService'),
    forumRolePermissionService = require('../../service/forumRolePermissionService'),
    forumPraiseLogService = require('../../service/forumPraiseLogService'),
    forumShareLogService = require('../../service/forumShareLogService'),
    forumCommentService = require('../../service/forumCommentService'),
    forumTagService = require('../../service/forumTagService'),
    forumUserService = require('../../service/forumUserService'),
    // var forumLimitActionRefService = require('../../service/forumLimitActionRefService');
    forumLimitOperationService = require('../../service/forumLimitOperationService'),
    forumAboutMEService = require('../../service/forumAboutMEService'),
    httpService = require('../../service/httpService'),
    redisThreadList = require('../../utils/redisThreadList');

var userUtil = require('../../utils/user');
var MemoryCache = require('../../utils/simpleCache');
var floorGenerator = require('../../utils/floorGenerator');
module.exports = function(app) {
    app.use('/admin/forum', router);
};


//---------------------文章表单操作---------------------------------
var getClientIP = function(req) {
    var ipAddress;
    var headers = req.headers;
    var forwardedIpsStr = headers['x-real-ip'] || headers['x-forwarded-for'];
    if (forwardedIpsStr) {
        ipAddress = forwardedIpsStr;
    } else {
        ipAddress = null;
    }
    if (!ipAddress) {
        ipAddress = req.connection.remoteAddress;
    }
    return ipAddress;
};

//格式化日期 (格式：年-月-日)
function date_format(date) {
    return moment(date).format('YYYY-MM-DD hh:mm:ss');
}

//新增论坛文章
router.post('/thread/create', function(req, res, next) {
    var rs = {},
        ent_code = req.session.user.ent_code,
        icon = req.session.user.headPic,
        nickName = req.session.user.nickName;
    if (icon && icon.indexOf('http://') == -1) {
        if (process.env.NODE_ENV === 'production') {
            icon = 'http://fs.wxpai.cn/' + icon;
        } else {
            icon = 'http://dev.fs.wxpai.cn/' + icon;
        }
    }
    req.body.ent_code = ent_code;
    if (!req.body.share) {
        delete req.body.share;
    }
    var uid = req.session.user.id;
    // var uid='12345';
    if (req.body.pid) {
        req.body.level = 2;
    }
    async.waterfall([
        function(callback) {
            forumUserService.getUserByUid(uid, callback);
        }
    ], function(err, user) {
        if (err) {
            res.json(returnCode.BUSY);
        } else {
            if (user) {
                //更新后台对应用户信息
                var isUpdate = false,
                    updateObj = {};
                if (icon) {
                    if (!user.icon || user.icon != icon) {
                        isUpdate = true;
                        updateObj.icon = icon;
                    }
                }
                if (nickName) {
                    if (!nickName || !user.nickName || user.nickName != nickName) {
                        isUpdate = true;
                        updateObj.nickName = nickName;
                    }
                }
                if (isUpdate) {
                    user.update(updateObj, function(err, data) {
                        if (err) console.log(err);
                    });
                }
                //创建帖子
                req.body.from = user._id;
                forumThreadService.createThread(req.body, function(err, entity) {
                    if (err) {
                        console.error(err);
                        res.json(returnCode.BUSY);
                    } else {
                        forumAboutMEService.saveThread(entity);
                        rs.data = {
                            'id': entity._id
                        };
                        res.json(_.assign(rs, returnCode.SUCCESS));
                    }
                });
            } else {
                //创建后台对应用户
                var entity = {
                    uid: uid,
                    ent_code: ent_code,
                    nickName: nickName,
                    icon: icon
                };
                forumUserService.createUser(entity, function(err, doc) {
                    if (err) {
                        res.json(returnCode.BUSY);
                    } else {
                        //创建帖子
                        req.body.from = doc._id;
                        forumThreadService.createThread(req.body, function(err, entity) {
                            if (err) {
                                console.error(err);
                                res.json(returnCode.BUSY);
                            } else {
                                forumAboutMEService.saveThread(entity);
                                rs.data = {
                                    'id': entity._id
                                };
                                res.json(_.assign(rs, returnCode.SUCCESS));
                            }
                        });
                    }
                });
            }
        }
    });
});

//获取目标论坛文章
router.get('/thread/:tid/get', function(req, res, next) {
    var uid = req.session.user.id;
    var tid = req.params.tid || null;
    var rs = {};
    if (tid) {
        async.waterfall([
            function(callback) {
                forumThreadService.getThreadById(tid, function(err, thread) {
                    if (err) {
                        callback(err, null);
                    } else {
                        callback(null, thread);
                    }
                });
            },
            function(thread, callback) {
                if (thread && thread.info && thread.info._id) {
                    forumTagService.getAllTag({
                        ent_code: req.session.user.ent_code,
                        info: thread.info._id
                    }, 1, 100, function(err, results) {
                        if (err) {
                            callback(err, null);
                        } else {
                            callback(null, [thread, results]);
                        }
                    });
                } else {
                    callback(null, [thread]);
                }

            },
            function(datas, callback) {
                forumUserService.getUserByUid(uid, function(err, results) {
                    if (err) {
                        callback(err, null);
                    } else {
                        datas.push(results)
                        callback(null, datas);
                    }
                });
            }
        ], function(err, results) {
            if (err) {
                res.json(returnCode.BUSY);
            } else {
                var isSameAuthor = false;
                if (results[0].from && results[0].from.uid) {
                    if (!isNaN(results[0].from.uid)) {
                        isSameAuthor = true;
                    }
                }

                if (results[0].pid) {
                    forumThreadService.getById(results[0].pid, function(err, parentThread) {
                        rs.data = results[0];
                        rs.tagList = results[1];
                        rs.parentThread = parentThread;
                        rs.isSameAuthor = isSameAuthor;
                        res.json(_.assign(rs, returnCode.SUCCESS));
                    });
                } else {
                    rs.data = results[0];
                    rs.tagList = results[1];
                    rs.isSameAuthor = isSameAuthor;
                    res.json(_.assign(rs, returnCode.SUCCESS));
                }

            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//更新文章状态、如：屏蔽
router.post('/thread/:tid/update', function(req, res, next) {
    var tid = req.params.tid;
    var fatherTitle = req.body.father;
    delete req.body.father;
    req.body.ent_code = req.session.user.ent_code;
    if (tid) {
        forumThreadService.updateThreadById(tid, req.body, function(err, thread) {
            if (err) {
                res.json(returnCode.BUSY);
            } else {
                if (!isNaN(req.body.status)) {
                    forumAboutMEService.updateThreadStatus(thread._id, thread.status)
                }
                res.json({
                    message: "success",
                    data: thread,
                    title: fatherTitle
                });
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//删除文章
router.post('/thread/:tid/delete', function(req, res, next) {
    var tid = req.params.tid;
    if (tid) {
        forumThreadService.deleteThreadById(tid, function(err, thread) {
            if (err) {
                res.json(returnCode.BUSY);
            } else {
                res.json(returnCode.SUCCESS);
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//逻辑删除文章
router.post('/thread/:tid/disable', function(req, res, next) {
    var tid = req.params.tid;
    if (tid) {
        forumThreadService.logicDeleteThreadById(tid, function(err, flag) {
            if (err) {
                res.json(returnCode.BUSY);
            } else {
                forumAboutMEService.updateThreadStatus(tid, 3);
                res.json(returnCode.SUCCESS);
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});


//文章置顶
router.post('/thread/:tid/:fid/top', function(req, res, next) {
    var tid = req.params.tid; //文章ID
    var fid = req.params.fid; //板块ID
    if (tid && fid) {
        forumThreadService.updateTopByThreadId(fid, tid, function(err, thread) {
            if (err) {
                res.json(returnCode.BUSY);
            } else {
                res.json(returnCode.SUCCESS);
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//文章取消置顶
router.post('/thread/:tid/:fid/unTop', function(req, res, next) {
    var tid = req.params.tid; //文章ID
    var fid = req.params.fid; //板块ID
    if (tid && fid) {
        forumThreadService.updateUnTopByThreadId(fid, tid, function(err, thread) {
            if (err) {
                res.json(returnCode.BUSY);
            } else {
                res.json(returnCode.SUCCESS);
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//文章加精
router.post('/thread/:tid/:fid/recommend', function(req, res, next) {
    var tid = req.params.tid,
        fid = req.params.fid,
        ent_code = req.session.user.ent_code,
        thread_from_id = req.body.thread_from_id,
        mid = req.body.mid;
    if (tid && fid) {
        forumThreadService.updateRecommendByThreadId(tid, function(err, thread) {
            if (err) {
                console.error(err);
                res.json(returnCode.BUSY);
            } else {
                if (mid) {
                    if(thread_from_id){
                        forumLimitOperationService.checkLimitOperationProhibitionAddIntegral(thread_from_id, function(err, flag) {
                            if (err) {
                                console.error(err);
                                res.json(returnCode.PROHIBITION_OF_SPEECH);
                            } else {
                                if (flag) {
                                    res.json(returnCode.PROHIBITION_OF_SPEECH);
                                } else {
                                    httpService.sendRequest(ent_code, mid, 'thread_recomment');
                                    res.json(returnCode.SUCCESS);
                                }
                            }
                        });
                    }else{
                        httpService.sendRequest(ent_code, mid, 'thread_recomment');
                        res.json(returnCode.SUCCESS);
                    }
                } else {
                    res.json(returnCode.SUCCESS);
                }
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//文章取消加精
router.post('/thread/:tid/:fid/unRecommend', function(req, res, next) {
    var tid = req.params.tid; //文章ID
    var fid = req.params.fid; //板块ID
    if (tid && fid) {
        forumThreadService.updateUnRecommendByThreadId(tid, function(err, thread) {
            if (err) {
                res.json(returnCode.BUSY);
            } else {
                res.json(returnCode.SUCCESS);
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//文章列表
router.get('/threads/list', function(req, res, next) {
    var pageNo = req.query.pageNo || 1;
    var pageSize = req.query.pageSize || 10;
    var infoId = req.query.infoId;
    var tagId = req.query.tagId;
    var pid = req.query.pid;
    var content = req.query.content;
    var type = req.query.type;
    var status = req.query.status;
    var nickName = req.query.nickName;
    var mid = req.query.mid;
    var quality = req.query.quality;
    var new_recommend = req.query.new_recommend;
    var eventStatus = req.query.eventStatus;
    var recommend_threads = req.query.recommend_threads;//推荐帖子
    var top_threads = req.query.top_threads;//置顶帖子
    var begin_time=req.query.begin_time;
    var end_time=req.query.end_time;
    var title = req.query.title;

    //update by Car 20160315 帖子管理增加查询条件查询后台发的帖
    var isPublishByBg = req.query.isPublishByBg;

    var sortBy = {
        top:-1,
        topTime:-1,
        quality:-1,
        created:-1
    };
    var conditions = {
        ent_code: req.session.user.ent_code,
        level: 1,
        status: {
            $ne: 3
        }
    };
    if (infoId) {
        conditions.info = infoId;
    }
    if (tagId) {
        conditions.tag = {
            $in: [tagId]
        };
        sortBy = {
            tag_top:-1,
            tag_topTime:-1,
            quality:-1,
            created:-1
        }
    }
    if (pid) {
        conditions.pid = pid;
        conditions.level = 2;
    }
    if (content) {
        conditions.content = {
            $regex: content,
            $options: 'i'
        }
    }
    if (type) {
        conditions.type = Number(type);
    }
    if (status) {
        conditions.status = Number(status);
    }
    if (quality) {
        conditions.quality = quality;
    }

    if (new_recommend) {
        conditions.new_recommend = new_recommend;
    }
    if(recommend_threads){//推荐贴
        conditions.new_recommend = 1;
        sortBy = {
            new_recommend_order_idx:-1,
            created:-1
        };
    }
    if(top_threads){//置顶帖
        conditions.top = 1;
        sortBy = {
            top_order_idx:-1,
            created:-1
        };
    }

    if (eventStatus) {
        var today = new Date();
        conditions.isEvent = 1;
        if(eventStatus == 0){
            conditions['event.eventStartTime'] = {$gt : today};
        }else if(eventStatus == 1){
            conditions['event.eventStartTime'] = {$lte : today},
            conditions['event.eventEndTime'] = {$gte : today}
        }else if(eventStatus == 2){
            conditions['event.eventEndTime'] = {$gt : today};
        }
    }

    if(begin_time && end_time){
        conditions.created = {$gte : begin_time, $lte : end_time};
    } else if(end_time){
        conditions.created = {$lte : end_time};
    } else if(begin_time){
        conditions.created = {$gte : begin_time};
    }

    if (title) {
        conditions.title = {
            $regex: title,
            $options: 'i'
        }
    }

    if(isPublishByBg){
        if(isPublishByBg == 1){
            conditions.isPublishByBg = isPublishByBg;
        }else if(isPublishByBg == 0){
            conditions.isPublishByBg = {$ne:1};
        }
    }



    if (mid) {
        forumThreadService.getAllThreadByFidAndMid(mid, conditions, pageNo, pageSize, sortBy, function(err, results) {
            if (err) {
                console.error(err);
                res.json(returnCode.BUSY);
            } else {
                if (pid) {
                    forumThreadService.getById(pid, function(err, parentThread) {
                        res.json(_.assign({
                            parentThread: parentThread
                        }, results, returnCode.SUCCESS));
                    });
                } else {
                    res.json(_.assign(results, returnCode.SUCCESS));
                }
            }
        });
    } else if (nickName) {
        forumThreadService.getAllThreadByFidAndNickName(nickName, conditions, pageNo, pageSize, sortBy, function(err, results) {
            if (err) {
                console.error(err);
                res.json(returnCode.BUSY);
            } else {

                if (pid) {
                    forumThreadService.getById(pid, function(err, parentThread) {
                        res.json(_.assign({
                            parentThread: parentThread
                        }, results, returnCode.SUCCESS));
                    });
                } else {
                    res.json(_.assign(results, returnCode.SUCCESS));
                }
            }
        });
    } else {
        forumThreadService.getAllThreadByFid(conditions, pageNo, pageSize, sortBy, function(err, results) {
            if (err) {
                console.error(err);
                res.json(returnCode.BUSY);
            } else {

                if (pid) {
                    forumThreadService.getById(pid, function(err, parentThread) {
                        res.json(_.assign({
                            parentThread: parentThread
                        }, results, returnCode.SUCCESS));
                    });
                } else {
                    res.json(_.assign(results, returnCode.SUCCESS));
                }
            }
        });
    }


});

//直接修改序号
function updateThreadTopIdx(id,idx,callback){
    // ForumThread.update({_id:id},{ top_order_idx: idx },
    //     null,
    //     function(err,result){
    //         if(err){
    //             console.error(err);
    //             callback(err,null);
    //         }else{
    //             callback(null,true);
    //         }
    //     });
    forumThreadService.updateThread({_id:id}, { top_order_idx: idx }, function(err, doc) {
        if(err){
            console.error(err);
            callback(err,null);
        }else{
            callback(null,true);
        }
    });
}
function updateThreadNewRecommendIdx(id,idx,callback){
    // ForumThread.update({_id:id},{ new_recommend_order_idx: idx },null, function(err,result){
    //     if(err){
    //         console.error(err);
    //         callback(err,null);
    //     }else{
    //         callback(null,true);
    //     }
    // });

    forumThreadService.updateThread({_id:id}, { new_recommend_order_idx: idx }, function(err, doc) {
        if(err){
            console.error(err);
            callback(err,null);
        }else{
            callback(null,true);
        }
    });
}
router.put('/top_threads/orderIDX/:id/:order', function(req, res, next) {
    var id=req.params.id;
    var idx=req.params.order;
    updateThreadTopIdx(id,idx,function(err,result){
        if(err ||　!result){
            console.log(err);
            return res.json(returnCode.BUSY);
        }
        return res.json(_.assign({data:true}, returnCode.SUCCESS));
    });
});
router.put('/recommend_threads/orderIDX/:id/:order', function(req, res, next) {
    var id=req.params.id;
    var idx=req.params.order;
    updateThreadNewRecommendIdx(id,idx,function(err,result){
        if(err ||　!result){
            console.log(err);
            return res.json(returnCode.BUSY);
        }
        return res.json(_.assign({data:true}, returnCode.SUCCESS));
    });
});
//评论列表
router.get('/thread/:tid/comment/list', function(req, res, next) {
    var tid = req.params.tid || null;
    var pageNo = req.query.pageNo || 1;
    var pageSize = req.query.pageSize || 10;
    var content = req.query.content || '';
    var conditions = {
        ent_code: req.session.user.ent_code,
        thread: tid,
        // level: '1',
        status: {
            $ne: 2
        } //过滤被删除的数据，status 为 2 表示删除
    };
    if (content) {
        conditions.content = {
            $regex: content,
            $options: 'i'
        };
    }
    if (tid) {
        //获取最新5条评论                                          
        forumCommentService.getCommentList(conditions, pageNo, pageSize, function(err, results) {
            if (err) {
                console.log(err);
                res.json(returnCode.BUSY);
            } else {
                var asyncTasks = [];
                _.forEach(results.items, function(e) {
                    asyncTasks.push(function(cb) {
                        if (e.level == 1) {
                            cb(null, e);
                        } else {
                            forumCommentService.getCommentParent(new mongoose.Types.ObjectId(e._id), function(err, p_comment) {
                                if(p_comment && p_comment.floor){
                                    var comment = e.toObject();
                                    comment.floor = p_comment.floor;
                                    cb(null, comment);
                                } else if(p_comment && !p_comment.floor){ //上一级没有楼层就再查上一级的评论
                                    var comment = e.toObject();
                                    comment.floor = comment.oneCommentFloor;
                                    cb(null, comment);
                                }else{
                                    cb(null, e);
                                }
                            });
                        }
                    });
                });
                async.parallel(asyncTasks, function(err, items) {
                    results.items = items;
                    res.json(_.assign(results, returnCode.SUCCESS));
                });
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//评论列表(根据楼层查询)
router.get('/thread/:tid/comment/list/:floor', function(req, res, next) {
    var tid = req.params.tid || null,
        floor = req.params.floor || null,
        content = req.query.content || '';
    var conditions = {
        ent_code: req.session.user.ent_code,
        thread: tid,
        level: '1',
        floor: floor,
        status: {
            $ne: 2
        } //过滤被删除的数据，status 为 2 表示删除
    };

    if (tid && floor) {
        forumCommentService.getAllComment(conditions, 1, 1, function(err, results) {
            if (err) {
                console.log(err);
                res.json(returnCode.BUSY);
            } else {
                var asyncTasks = [];
                if (results.items && results.items[0] && results.items[0].comments.length > 0) {
                    var items = [results.items[0]];
                    _.forEach(results.items[0].comments, function(e) {
                        var subComment = e.toObject();
                        subComment.floor = results.items[0].floor;
                        if (content) {
                            if (subComment.content.indexOf(content) != -1) {
                                items.push(subComment);
                            }
                        } else {
                            items.push(subComment);
                        }
                    });
                    results.items = items;
                    results.total = items.length;
                    res.json(_.assign(results, returnCode.SUCCESS));
                } else {
                    res.json(_.assign(results, returnCode.SUCCESS));
                }
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//评论列表(根据楼层查询)
router.get('/thread/:tid/comment/list/member/:mid', function(req, res, next) {
    var tid = req.params.tid || null,
        content = req.query.content || '',
        mid = req.params.mid,
        pageNo = req.query.pageNo || 1,
        pageSize = req.query.pageSize || 10;
    var conditions = {
        ent_code: req.session.user.ent_code,
        thread: tid,
        status: {
            $ne: 2
        } //过滤被删除的数据，status 为 2 表示删除
    };
    if (content) {
        conditions.content = {
            $regex: content,
            $options: 'i'
        };
    }
    if (tid && mid) {
        forumCommentService.getCommentListByMid(mid, conditions, pageNo, pageSize, function(err, results) {
            if (err) {
                console.error(err);
                res.json(returnCode.BUSY);
            } else {
                res.json(_.assign(results, returnCode.SUCCESS));
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//查询评论列表
router.get('/thread/:tid/comment/listV2', function(req, res, next) {
    var tid = req.params.tid || null,
        pageNo = req.query.pageNo || 1,
        pageSize = req.query.pageSize || 10,
        mid = req.query.mid,
        floor = req.query.floor,
        content = req.query.content || '';
    var conditions = {
        ent_code: req.session.user.ent_code,
        thread: tid
    };
    if (content) {
        conditions.content = {
            $regex: content,
            $options: 'i'
        };
    }

    if(floor){
        conditions.level = '1';
        conditions.floor = floor;
    }

    if (tid && mid) {
        forumCommentService.getCommentListByMid(mid, conditions, pageNo, pageSize, function(err, results) {
            if (err) {
                console.error(err);
                res.json(returnCode.BUSY);
            } else {
                var asyncTasks = [];
                if (results.items && results.items[0] && results.items[0].comments.length > 0) {
                    var items = [results.items[0]];
                    _.forEach(results.items[0].comments, function(e) {
                        var subComment = e.toObject();
                        subComment.floor = results.items[0].floor;
                        if (content) {
                            if (subComment.content.indexOf(content) != -1) {
                                items.push(subComment);
                            }
                        } else {
                            items.push(subComment);
                        }
                    });
                    results.items = items;
                    results.total = items.length;
                    res.json(_.assign(results, returnCode.SUCCESS));
                } else {
                    res.json(_.assign(results, returnCode.SUCCESS));
                }
            }
        });
    } else if(tid && floor){
        forumCommentService.getAllComment(conditions, 1, 1, function(err, results) {
            if (err) {
                console.log(err);
                res.json(returnCode.BUSY);
            } else {
                var asyncTasks = [];
                if (results.items && results.items[0] && results.items[0].comments.length > 0) {
                    var items = [results.items[0]];
                    _.forEach(results.items[0].comments, function(e) {
                        var subComment = e.toObject();
                        subComment.floor = results.items[0].floor;
                        if (content) {
                            if (subComment.content.indexOf(content) != -1) {
                                items.push(subComment);
                            }
                        } else {
                            items.push(subComment);
                        }
                    });
                    results.items = items;
                    results.total = items.length;
                    res.json(_.assign(results, returnCode.SUCCESS));
                } else {
                    res.json(_.assign(results, returnCode.SUCCESS));
                }
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});


// 更新评论状态
router.post('/thread/comment/update/:cid', function(req, res, next) {
    var cid = req.params.cid || null;
    var status = req.body.status;
    if (cid) {
        async.waterfall([
            function(callback) {
                //获取评论实例
                forumCommentService.getCommentById(cid, callback);
            },
            function(comment, callback) {
                //更新评论状态
                forumCommentService.updateCommentStatusById(cid, status, function(err, update) {
                    callback(err, comment);
                });
            },
            function(comment, callback) {
                //是否删除评论
                if (status == 2) {
                    //一级评论
                    if (comment.level == 1) {
                        var commentList = comment.comments;
                        var comments = forumThreadService.remove(commentList, cid);
                        forumThreadService.updateThreadById(comment.thread, {
                            comments: comments,
                            $inc: {
                                comment_count: -1
                            }
                        }, function(err, result) {
                            callback(err);
                        });
                    } else {
                        forumThreadService.updateThreadCommentCount(comment.thread, function(err, result) {
                            if (err) {
                                callback(err);
                            } else {
                                // 更新评论的子评论列表
                                forumCommentService.getCommentParent(cid, function(err, p_comment) {
                                    if (err || !p_comment) {
                                        callback(err || 'comment not exist');
                                    } else {
                                        var commentList = p_comment.comments;
                                        var comments = forumCommentService.remove(commentList, cid);
                                        forumCommentService.updateCommentById(p_comment._id, {
                                            comments: comments
                                        }, function(err, update) {
                                            callback(err);
                                        });
                                    }
                                });
                            }
                        });
                    }
                } else {
                    if (comment && comment.level == 1) {
                        forumAboutMEService.updateCommentLevel1Status(comment._id, status);
                    }
                    if (comment && comment.level == 2) {
                        forumAboutMEService.updateCommentLevel2Status(comment._id, status);
                    }
                    callback();
                }
            }
        ], function(err, restult) {
            if (err) {
                console.error(err);
                res.json(returnCode.BUSY);
            } else {
                res.json(returnCode.SUCCESS);
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

// 更新回复评论
router.post('/thread/comment/update/:cid/comments', function(req, res, next) {
    var tid = req.body.tid || null;
    var cid = req.params.cid || null;
    var floor = req.body.floor || null;
    var replayComment_id = req.body.replayComment_id;
    var ent_code = req.session.user.ent_code;
    var conditions = {
        "thread": tid,
        "level" : "1",
        "floor" : floor,
        "ent_code" : ent_code
    };

    forumCommentService.updateComment(conditions, {
        $push: {
            comments: replayComment_id
        },
        $inc: {
            comment_count: 1
        }
    }, function(err, result) {
        if (err) {
            console.error(err);
            res.json(returnCode.BUSY);
        } else {
            forumAboutMEService.saveCommentLevel2BySearch(tid, cid, replayComment_id);
            res.json(returnCode.SUCCESS);
        }
    });
});

//添加评论
router.post('/thread/comment/add', function(req, res, next) {
    var tid = req.body.tid || null;
    var entity = req.body;
    entity.created = new Date();
    entity.ent_code = req.session.user.ent_code;
    entity.ip = getClientIP(req);

    var uid = req.session.user.id;
    if (tid) {
        async.waterfall([
            function(callback) {
                forumUserService.getUserByUid(uid, callback);
            },
            function(user, callback) {
                //获取发表人
                if (!user) {
                    var userentity = {
                        uid: uid,
                        nickName: req.session.user.name,
                        icon: req.session.user.headPic
                    };
                    forumUserService.createUser(userentity, function(err, doc) {
                        if (err) {
                            callback(err);
                        } else {
                            entity.from = doc._id;
                            callback();
                        }
                    });
                } else {
                    var icon = req.session.user.headPic;
                    if (icon) {
                        if (icon.indexOf('http://') == -1) {
                            icon = req.host != 'piplus.wxpai.cn' ? 'http://img3.wxpai.cn/' + icon : 'http://img2.wxpai.cn/' + icon;
                        }
                        if (icon && (!user.icon || user.icon !== icon)) {

                            forumUserService.updateUserById(user._id, {
                                icon: icon
                            }, function(err) {
                                if (err) {
                                    console.log(err);
                                }
                            });
                        }
                    }
                    entity.from = user;
                    callback();
                }
            },
            function(callback) {
                //获取楼层
                if (entity.level == 1) {
                    floorGenerator.getCommentFloor(entity.ent_code,tid,function(err,floor){
                        if (err) {
                            callback(err);
                        } else {
                            entity.floor = floor;
                            callback();
                        }
                    });
                } else {
                    callback();
                }
            },
            function(callback) {
                // 添加评论
                forumCommentService.createComment(entity, function(err, comment) {
                    if (err) {
                        callback(err);
                    } else if(entity.level == 1){
                        forumThreadService.getById(tid, function(err, thread) {
                            if (entity.level == 1) {
                                forumAboutMEService.saveCommentLevel1(thread, comment);
                            }
                            var comments = thread.comments;
                            var array = [];
                            if (comments && comments.length > 0) {
                                array = comments;
                            }
                            array.push(comment._id);
                            forumThreadService.updateThreadById(tid, {
                                comments: array,
                                $inc: {
                                    comment_count: 1
                                }
                            }, function(err, result) {
                                var returnData = {
                                    comment: comment,
                                    errorcode: 0,
                                    errormsg: '请求成功'
                                }
                                callback(null, returnData);
                            });
                        });
                    }else{
                        var returnData = {
                            comment: comment,
                            errorcode: 0,
                            errormsg: '请求成功'
                        }
                        callback(null, returnData);
                    }
                });
            }
        ], function(err, returnData) {
            if (err) {
                console.error(err);
                res.json(returnCode.BUSY);
            } else {
                res.json(returnData);
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

// 更新文章评论
router.post('/thread/:tid/updateComments', function(req, res, next) {
    var tid = req.params.tid || null;
    var cid = req.body.cid || null;

    forumThreadService.updateThreadById(tid, {
            $push: {
                comments: cid
            },
            $inc: {
                comment_count: 1
            }
        },
        function(err, result) {
            if (err) {
                console.error(err);
                res.json(returnCode.BUSY);
            } else {
                res.json(returnCode.SUCCESS);
            }
        });

    // 获取评论的子评论列表
    // forumThreadService.getById(tid, function(err, result) {
    //     if (err) {
    //         console.error(err);
    //         res.json(returnCode.BUSY);
    //     } else {
    //         result.comments.push(cid);
    //         result.comment_count += 1;

    //         forumThreadService.updateThreadById(tid, result,
    //             function(err, result) {
    //                 if (err) {
    //                     console.error(err);
    //                     res.json(returnCode.BUSY);
    //                 } else {
    //                     res.json(returnCode.SUCCESS);
    //                 }
    //             });
    //     }
    // });
});

var checkOpenIds = function(openIds, openId) {
    var restult = false;
    for (var i = 0; i < openIds.length; i += 1) {
        if (openIds[i] == openId) {
            restult = true;
            break;
        }
    }
    return restult;
};
var getSubSpreadPaths = function(openIds, c, pvs, cb) {
    c.children = [];
    for (var i = 0; i < pvs.length; i += 1) {
        if (pvs[i].source.uid == c.openId) {
            var flag = false;
            for (var j = 0; j < c.children.length; j += 1) {
                if (c.children[j].openId == pvs[i].open_id) {
                    flag = true;
                    break;
                }
            }
            if (!flag && !checkOpenIds(openIds, pvs[i].open_id)) {
                c.children.push({
                    openId: pvs[i].open_id,
                    nickName: pvs[i].user.displayName || '未知',
                    name: pvs[i].user.nickName || '未知' , 
                    id : pvs[i]._id
                });
                openIds.push(pvs[i].open_id);
            }
        }
    }
    var new_pvs = [];
    for (var i = 0; i < pvs.length; i += 1) {
        if (pvs[i].source.uid !== c.openId) {
            new_pvs.push(pvs[i]);
        }
    }
    pvs = new_pvs;
    if (c.children.length > 0) {
        var thenTask = [];
        _.forEach(c.children, function(c) {
            thenTask.push(function(scb) {
                getSubSpreadPaths(openIds, c, pvs, scb);
            });
        })
        then.parallel(thenTask).then(function(cont, childrens) {
            c.children = childrens;
            cb(null, c);
        });
    } else {
        cb(null, c);
    }
};
// 更新文章评论
router.get('/thread/:tid/spreadchain', function(req, res, next) {
    var tid = req.params.tid || null,
        ent_code = req.session.user.ent_code;
    var openIds = [];
    then.parallel([
        function(cont) {
            ForumThread.findOne({
                _id: tid,
                ent_code: ent_code
            }).populate('from').exec(function(err, doc) {
                cont(err, doc);
            });
        },
        function(cont) {
            ForumPVLog.find({
                thread: tid,
                ent_code: ent_code
            }).sort('source created').populate('user source').exec(function(err, docs) {
                cont(err, docs);
            });
        }
    ]).then(function(cont, results) {
        var thread = results[0],
            pvs = results[1];

        var data = {
                openId: thread.from.uid,
                name: thread.from.nickName,
                //nickName: thread.from.displayName,
                nickName: thread.from.nickName,
                children: [] , 
                id : 'ROOT'
            }
            //获取一级传播
        for (var i = 0; i < pvs.length; i += 1) {
            var pv = pvs[i];
            if (!pv.source) {
                var flag = false;
                for (var j = 0; j < data.children.length; j += 1) {
                    var c = data.children[j];
                    if (c.openId === pv.open_id) {
                        flag = true;
                        break;
                    }
                }
                if (!flag && pv.open_id !== data.openId && !checkOpenIds(openIds, pv.open_id)) {
                    data.children.push({
                        openId: pv.open_id,
                        nickName: pv.user.displayName || '未知',
                        name: pv.user.nickName || '未知',
                        id : pv._id
                    });
                    openIds.push(pv.open_id);
                }
            }
        }
        //获取非一级传播
        var new_pvs = [];
        for (var i = 0; i < pvs.length; i += 1) {
            if (pvs[i].source && pvs[i].source.uid !== data.openId && pvs[i].open_id !== data.openId) {
                new_pvs.push(pvs[i]);
            }
        }
        pvs = new_pvs;
        //递归其他传播
        var thenTask = [];
        _.forEach(data.children, function(c) {
            thenTask.push(function(cb) {
                getSubSpreadPaths(openIds, c, pvs, cb);
            });
        })
        then.parallel(thenTask).then(function(cont, childrens) {
            data.children = childrens;
            var rs = {
                data: data
            }
            openIds = [];
            res.json(_.assign(rs, returnCode.SUCCESS));
        });

    }).fail(function(cont, err) {
        console.error(err);
        res.json(returnCode.BUSY);
    });
});


//udpate by Car 20151208 使用redis优化传播链图
router.get('/thread/:tid/spreadchainByRedis', function(req, res, next) {
    var tid = req.params.tid || null,
        ent_code = req.session.user.ent_code;
    var key = "sc_forum_thread_" + tid;
    redis.get(key,function(err,store){
        if(store){
            return res.json(_.assign({
                data: key
            }, returnCode.SUCCESS));
        }
        then.parallel([
            function(cont) {
                ForumThread.findOne({
                    _id: tid,
                    ent_code: ent_code
                }).select('_id from').populate({
                    path:'from',
                    select:'nickName uid'
                }).exec(function(err, doc) {
                    cont(err,doc);
                });
            },
            function(cont) {
                var conditions = {
                    ent_code: ent_code,
                    thread: tid
                };
                var s1 = new Date().getTime();
                ForumPVLog.count(conditions).exec(function(err, count) {
                    var pageSize = 5000;
                    var page_total = Math.ceil((count || 0) / pageSize) + 1;
                    var tasks = [];
                    _.each(_.range(page_total),function(pageNo){
                        tasks.push(function(cb){
                            ForumPVLog.find(conditions).select('open_id user source').sort('source created')
                                .skip((pageNo)*pageSize).limit(pageSize)
                                .populate({
                                    path: 'user source',
                                    select: 'uid nickName displayName'
                                }).exec(function(err, docs) {
                                    cb(err,docs);
                                });
                        });
                    });
                    async.parallel(tasks,function(err,results){
                        console.log('加载数据耗时',(new Date().getTime()-s1)/1000);
                        cont(err,_.flatten(results));
                    });
                });
            }
        ]).then(function(cont, results) {
            var thread = results[0],
                pvs = results[1];
            var datas = [];
            datas.push({
                openId: thread.from.uid,
                nickName: thread.from.nickName || '未知',
                name: thread.from.nickName || '未知',
                parent:null
            });
            _.forEach(pvs,function(pv){
                datas.push({
                    openId: pv.open_id,
                    nickName: (pv.user && pv.user.displayName) || '未知',
                    name: (pv.user && pv.user.nickName) || '未知',
                    parent:(pv.source && pv.source.uid) || thread.from.uid
                });
            });
            redis.set(key,JSON.stringify(datas));
            redis.expire(key,60 * 30);
            res.json(_.assign({
                data: key
            }, returnCode.SUCCESS));
        }).fail(function(cont, err) {
            console.error(err);
            res.json(returnCode.BUSY);
        });
    });
});


// 导出文章评论内容
router.get('/thread/:tid/exportComments', function(req, res, next) {
    var tid = req.params.tid || null,
        ent_code = req.session.user.ent_code || null,
        mid = req.query.mid,
        floor = req.query.floor,
        content = req.query.content;
    then(function(cont) {
        if (tid && ent_code) {
            then.parallel([
                function(cont2) {
                    forumThreadService.getThreadByIdSimple({
                        _id: tid
                    }, function(err, doc) {
                        forumPraiseLogService.queryPraiseLogV2({
                            ent_code: ent_code,
                            thread: tid,
                            type: 1
                        }, function(err, results) {
                            if (err) {
                                cont2(err);
                            } else {
                                if (results && results.length > 0) {
                                    var praiseNames = [];
                                    var praiseOpenIds = [];
                                    _.forEach(results, function(e) {
                                        if (e.user) {
                                            praiseNames.push(e.user.nickName || '游客');
                                            praiseOpenIds.push(e.user.uid);
                                        }
                                    });
                                    doc = doc.toObject();
                                    doc.praiseNames = praiseNames.toString();
                                    doc.praiseOpenIds = praiseOpenIds.toString();
                                    cont2(null, doc);
                                } else {
                                    doc = doc.toObject();
                                    doc.praiseNames = '';
                                    doc.praiseOpenIds = '';
                                    cont2(null, doc);
                                }
                            }
                        });
                    });
                },
                function(cont2) {
                    var conditions = {
                        ent_code: ent_code,
                        thread: tid
                    };
                    if (content) {
                        conditions.content = {
                            $regex: content,
                            $options: 'i'
                        };
                    }
                    if (mid) {
                        forumCommentService.getCommentListByMid(mid, conditions, 1, 10000, function(err, results) {
                            cont2(err, results ? results.items : []);
                        });
                    } else if (floor) {
                        conditions.level = '1';
                        conditions.floor = floor;
                        forumCommentService.getAllComment(conditions, 1, 1, function(err, results) {
                            cont2(err, results ? results.items : []);
                        });
                    } else {
                        conditions.level = '1';
                        //查询所有评论
                        forumCommentService.getAllCommentByThread(conditions, function(err, docs) {
                            cont2(err, docs);
                        });
                    }

                }
            ]).then(function(cont2, datas) {
                cont(null, datas[0], datas[1]);
            }).fail(function(cont2, err) {
                cont(err);
            });

        } else {
            cont('params error');
        }
    }).then(function(cont, thread, comments) {
        //获取点赞信息
        var asyncTasks = [];
        _.forEach(comments, function(e) {
            asyncTasks.push(function(cb) {
                var conditions = {
                    ent_code: ent_code,
                    thread: tid,
                    type: 2,
                    comment: e._id
                }
                forumPraiseLogService.queryPraiseLogV2(conditions, function(err, results) {
                    if (err) {
                        cb(null, e);
                    } else {
                        if (results) {
                            var praiseNames = [];
                            var praiseOpenIds = [];
                            _.forEach(results, function(e2) {
                                if (e2.user) {
                                    praiseNames.push(e2.user.nickName || '游客');
                                    praiseOpenIds.push(e2.user.uid);
                                }
                            });
                            e.praiseNames = praiseNames.toString();
                            e.praiseOpenIds = praiseOpenIds.toString();
                            e.praiseCount = praiseNames.length;
                        }
                        cb(null, e);
                    }
                });
            });
        });
        async.parallel(asyncTasks, function(err, items) {
            cont(err, thread, items);
        });
    }).then(function(cont, thread, items) {
        //表头
        var conf = {},
            rows = [];
        conf.cols = [{
            caption: '帖子标题',
            type: 'string'
        }, {
            caption: '发布时间',
            type: 'string'
        }, {
            caption: '发帖人',
            type: 'string'
        }, {
            caption: 'openId',
            type: 'string'
        }, {
            caption: '会员编号',
            type: 'string'
        }, {
            caption: '内容',
            type: 'string'
        }, {
            caption: '评论数',
            type: 'string'
        }, {
            caption: '转发数',
            type: 'string'
        }, {
            caption: '点赞数',
            type: 'string'
        }, {
            caption: '点赞人',
            type: 'string'
        }, {
            caption: '点赞点赞人openId',
            type: 'string'
        }];
        var contentReplace = function(str) {
                var strAfter = '';
                var regex = /[\0-\x1F\x7F-\x9F\xAD\u0378\u0379\u037F-\u0383\u038B\u038D\u03A2\u0528-\u0530\u0557\u0558\u0560\u0588\u058B-\u058E\u0590\u05C8-\u05CF\u05EB-\u05EF\u05F5-\u0605\u061C\u061D\u06DD\u070E\u070F\u074B\u074C\u07B2-\u07BF\u07FB-\u07FF\u082E\u082F\u083F\u085C\u085D\u085F-\u089F\u08A1\u08AD-\u08E3\u08FF\u0978\u0980\u0984\u098D\u098E\u0991\u0992\u09A9\u09B1\u09B3-\u09B5\u09BA\u09BB\u09C5\u09C6\u09C9\u09CA\u09CF-\u09D6\u09D8-\u09DB\u09DE\u09E4\u09E5\u09FC-\u0A00\u0A04\u0A0B-\u0A0E\u0A11\u0A12\u0A29\u0A31\u0A34\u0A37\u0A3A\u0A3B\u0A3D\u0A43-\u0A46\u0A49\u0A4A\u0A4E-\u0A50\u0A52-\u0A58\u0A5D\u0A5F-\u0A65\u0A76-\u0A80\u0A84\u0A8E\u0A92\u0AA9\u0AB1\u0AB4\u0ABA\u0ABB\u0AC6\u0ACA\u0ACE\u0ACF\u0AD1-\u0ADF\u0AE4\u0AE5\u0AF2-\u0B00\u0B04\u0B0D\u0B0E\u0B11\u0B12\u0B29\u0B31\u0B34\u0B3A\u0B3B\u0B45\u0B46\u0B49\u0B4A\u0B4E-\u0B55\u0B58-\u0B5B\u0B5E\u0B64\u0B65\u0B78-\u0B81\u0B84\u0B8B-\u0B8D\u0B91\u0B96-\u0B98\u0B9B\u0B9D\u0BA0-\u0BA2\u0BA5-\u0BA7\u0BAB-\u0BAD\u0BBA-\u0BBD\u0BC3-\u0BC5\u0BC9\u0BCE\u0BCF\u0BD1-\u0BD6\u0BD8-\u0BE5\u0BFB-\u0C00\u0C04\u0C0D\u0C11\u0C29\u0C34\u0C3A-\u0C3C\u0C45\u0C49\u0C4E-\u0C54\u0C57\u0C5A-\u0C5F\u0C64\u0C65\u0C70-\u0C77\u0C80\u0C81\u0C84\u0C8D\u0C91\u0CA9\u0CB4\u0CBA\u0CBB\u0CC5\u0CC9\u0CCE-\u0CD4\u0CD7-\u0CDD\u0CDF\u0CE4\u0CE5\u0CF0\u0CF3-\u0D01\u0D04\u0D0D\u0D11\u0D3B\u0D3C\u0D45\u0D49\u0D4F-\u0D56\u0D58-\u0D5F\u0D64\u0D65\u0D76-\u0D78\u0D80\u0D81\u0D84\u0D97-\u0D99\u0DB2\u0DBC\u0DBE\u0DBF\u0DC7-\u0DC9\u0DCB-\u0DCE\u0DD5\u0DD7\u0DE0-\u0DF1\u0DF5-\u0E00\u0E3B-\u0E3E\u0E5C-\u0E80\u0E83\u0E85\u0E86\u0E89\u0E8B\u0E8C\u0E8E-\u0E93\u0E98\u0EA0\u0EA4\u0EA6\u0EA8\u0EA9\u0EAC\u0EBA\u0EBE\u0EBF\u0EC5\u0EC7\u0ECE\u0ECF\u0EDA\u0EDB\u0EE0-\u0EFF\u0F48\u0F6D-\u0F70\u0F98\u0FBD\u0FCD\u0FDB-\u0FFF\u10C6\u10C8-\u10CC\u10CE\u10CF\u1249\u124E\u124F\u1257\u1259\u125E\u125F\u1289\u128E\u128F\u12B1\u12B6\u12B7\u12BF\u12C1\u12C6\u12C7\u12D7\u1311\u1316\u1317\u135B\u135C\u137D-\u137F\u139A-\u139F\u13F5-\u13FF\u169D-\u169F\u16F1-\u16FF\u170D\u1715-\u171F\u1737-\u173F\u1754-\u175F\u176D\u1771\u1774-\u177F\u17DE\u17DF\u17EA-\u17EF\u17FA-\u17FF\u180F\u181A-\u181F\u1878-\u187F\u18AB-\u18AF\u18F6-\u18FF\u191D-\u191F\u192C-\u192F\u193C-\u193F\u1941-\u1943\u196E\u196F\u1975-\u197F\u19AC-\u19AF\u19CA-\u19CF\u19DB-\u19DD\u1A1C\u1A1D\u1A5F\u1A7D\u1A7E\u1A8A-\u1A8F\u1A9A-\u1A9F\u1AAE-\u1AFF\u1B4C-\u1B4F\u1B7D-\u1B7F\u1BF4-\u1BFB\u1C38-\u1C3A\u1C4A-\u1C4C\u1C80-\u1CBF\u1CC8-\u1CCF\u1CF7-\u1CFF\u1DE7-\u1DFB\u1F16\u1F17\u1F1E\u1F1F\u1F46\u1F47\u1F4E\u1F4F\u1F58\u1F5A\u1F5C\u1F5E\u1F7E\u1F7F\u1FB5\u1FC5\u1FD4\u1FD5\u1FDC\u1FF0\u1FF1\u1FF5\u1FFF\u200B-\u200F\u202A-\u202E\u2060-\u206F\u2072\u2073\u208F\u209D-\u209F\u20BB-\u20CF\u20F1-\u20FF\u218A-\u218F\u23F4-\u23FF\u2427-\u243F\u244B-\u245F\u2700\u2B4D-\u2B4F\u2B5A-\u2BFF\u2C2F\u2C5F\u2CF4-\u2CF8\u2D26\u2D28-\u2D2C\u2D2E\u2D2F\u2D68-\u2D6E\u2D71-\u2D7E\u2D97-\u2D9F\u2DA7\u2DAF\u2DB7\u2DBF\u2DC7\u2DCF\u2DD7\u2DDF\u2E3C-\u2E7F\u2E9A\u2EF4-\u2EFF\u2FD6-\u2FEF\u2FFC-\u2FFF\u3040\u3097\u3098\u3100-\u3104\u312E-\u3130\u318F\u31BB-\u31BF\u31E4-\u31EF\u321F\u32FF\u4DB6-\u4DBF\u9FCD-\u9FFF\uA48D-\uA48F\uA4C7-\uA4CF\uA62C-\uA63F\uA698-\uA69E\uA6F8-\uA6FF\uA78F\uA794-\uA79F\uA7AB-\uA7F7\uA82C-\uA82F\uA83A-\uA83F\uA878-\uA87F\uA8C5-\uA8CD\uA8DA-\uA8DF\uA8FC-\uA8FF\uA954-\uA95E\uA97D-\uA97F\uA9CE\uA9DA-\uA9DD\uA9E0-\uA9FF\uAA37-\uAA3F\uAA4E\uAA4F\uAA5A\uAA5B\uAA7C-\uAA7F\uAAC3-\uAADA\uAAF7-\uAB00\uAB07\uAB08\uAB0F\uAB10\uAB17-\uAB1F\uAB27\uAB2F-\uABBF\uABEE\uABEF\uABFA-\uABFF\uD7A4-\uD7AF\uD7C7-\uD7CA\uD7FC-\uF8FF\uFA6E\uFA6F\uFADA-\uFAFF\uFB07-\uFB12\uFB18-\uFB1C\uFB37\uFB3D\uFB3F\uFB42\uFB45\uFBC2-\uFBD2\uFD40-\uFD4F\uFD90\uFD91\uFDC8-\uFDEF\uFDFE\uFDFF\uFE1A-\uFE1F\uFE27-\uFE2F\uFE53\uFE67\uFE6C-\uFE6F\uFE75\uFEFD-\uFF00\uFFBF-\uFFC1\uFFC8\uFFC9\uFFD0\uFFD1\uFFD8\uFFD9\uFFDD-\uFFDF\uFFE7\uFFEF-\uFFFB\uFFFE\uFFFF]/g
                if (str) {
                    strAfter = str.replace(/<[^>]+>/g, '').replace(regex, '');
                }
                return strAfter;
            }
            //文章内容
        var t_title = thread.title == '发话题' ? '无标题' : thread.title,
            t_time = date_format(thread.created),
            t_from = contentReplace(thread.from ? thread.from.nickName : '游客'),
            t_from_openId = thread.from ? thread.from.uid : '暂无',
            t_praiseNames = contentReplace(thread.praiseNames),
            t_praiseOpenIds = thread.praiseOpenIds,
            t_from_mid = thread.from ? thread.from.mid : '暂无';


        rows.push([
            t_title,
            t_time,
            t_from,
            t_from_openId,
            t_from_mid ? t_from_mid.toString() : '暂无',
            contentReplace(thread.content),
            thread.comment_count.toString(),
            thread.share_count.toString(),
            thread.praise_count.toString(),
            t_praiseNames,
            t_praiseOpenIds
        ]);
        rows.push(['', '', '', '', '', '', '', '', '', '', '']);
        rows.push(['楼层', '发布时间', '会员名', 'openId', '会员编号', '内容', '点赞数', '点赞人', '点赞人openId', null, null]);
        for (var i = items.length - 1; i >= 0; i--) {
            var comment = items[i];
            var thisFloor = comment.floor || '互动',
                time = date_format(comment.created),
                openId = '暂无',
                nickName = '暂无',
                fromMid = '暂无',
                content = comment.content ? contentReplace(comment.content) : '暂无',
                praiseCount = comment.praiseCount || 0,
                praiseNames = '暂无',
                praiseOpenIds = '暂无';
            if (comment.praiseNames) {
                praiseNames = comment.praiseNames.toString();
                praiseOpenIds = comment.praiseOpenIds.toString();
            }
            if (comment.from) {
                openId = comment.from.uid || '暂无';
                nickName = comment.from.displayName || comment.from.nickName || '游客';
                fromMid = comment.from.mid || '暂无';
            }

            var this_row = [
                thisFloor + '',
                time,
                contentReplace(nickName),
                openId,
                fromMid.toString(),
                content,
                praiseCount.toString(),
                praiseNames,
                praiseOpenIds,
                null, null
            ];
            rows.push(this_row);
            if (!mid && comment.comments.length > 0) {
                for (var k = 0; k < comment.comments.length; k++) {
                    var subComment = comment.comments[k];
                    var subTime = date_format(subComment.created),
                        subOpenId = '暂无',
                        subMid = '暂无',
                        subNickName = '暂无',
                        subContent = subComment.content ? contentReplace(subComment.content) : '暂无',
                        subTo = '暂无';
                    if (subComment.from) {
                        subOpenId = subComment.from.uid || '暂无';
                        subNickName = subComment.from.nickName || '游客';
                        subMid = subComment.from.mid || '暂无';
                    }
                    if (subComment.to) {
                        subTo = contentReplace(subComment.to.nickName) || '游客';
                    }
                    var subRow = [
                        subNickName + ' 回复 ' + subTo,
                        subTime,
                        subNickName,
                        subOpenId,
                        subMid.toString(),
                        content,
                        '0',
                        null, null, null, null
                    ];
                    rows.push(subRow);
                };
            }
        };
        conf.rows = rows;
        var result = nodeExcel.execute(conf);
        res.charset = 'utf-8';
        res.setHeader('Content-Type', 'application/vnd.openxmlformats');
        res.setHeader('Content-Disposition', 'attachment; filename=commentsReport.xlsx');
        res.end(result, 'binary');
    }).fail(function(cont, err) {
        console.error(err);
        if (err === 'params error') {
            res.json(returnCode.WRONG_PARAM);
        } else {
            res.json(returnCode.BUSY);
        }
    });
});

//更新文章质量
router.post('/thread/:tid/quality', function(req, res, next) {
    var tid = req.params.tid; //文章ID
    var quality = req.body.quality;
    if (tid && quality) {
        forumThreadService.updateThread({
            _id: tid
        }, {
            quality: quality
        }, function(err, thread) {
            if (err) {
                res.json(returnCode.BUSY);
            } else {
                res.json(returnCode.SUCCESS);
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});


//文章推荐
router.post('/thread/:tid/essence', function(req, res, next) {
    var tid = req.params.tid
    var ent_code = req.session.user.ent_code;
    if (tid) {
        var condition = {
            _id: tid,
            ent_code: ent_code
        };
        var entity = {
            new_recommend: 1,
            new_recommend_time: new Date()
        };

        forumThreadService.updateThread(condition, entity, function(err, doc) {
            if (err) {
                console.error(err);
                res.json(returnCode.BUSY);
            } else {
                res.json(returnCode.SUCCESS);
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//文章取消推荐
router.post('/thread/:tid/unEssence', function(req, res, next) {
    var tid = req.params.tid; //文章ID
    var ent_code = req.session.user.ent_code;
    if (tid) {
        var condition = {
            _id: tid,
            ent_code: ent_code
        };
        var entity = {
            new_recommend: 0,
            new_recommend_time: null
        };
        forumThreadService.updateThread(condition, entity, function(err, doc) {
            if (err) {
                console.error(err);
                res.json(returnCode.BUSY);
            } else {
                res.json(returnCode.SUCCESS);
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});





//H5创建子话题文章
router.post('/h5/thread/create', function(req, res, next) {
    var pid = req.body.pid;
    var title = req.body.title;
    var content = req.body.content;
    var images = req.body.images;
    var ent_code = req.body.ent_code;
    var nickName = req.body.nickName;
    var uid = req.body.openId;
    var mid = req.body.mid;
    var icon = req.body.icon;

    var thread_entity = {
        pid : pid,
        title : title,
        content : content,
        ent_code: ent_code,
        images: images,
        type : 1,
        level : 2
    }

    if(pid && title && content && ent_code && uid && mid){
        async.waterfall([
            function(callback) {
                forumUserService.getUserByUid(uid, callback);
            },
            function(user, callback) { //创建用户
                if (user) {
                    callback(null, user);
                } else {
                    var entity = {
                        mid: mid,
                        uid: uid,
                        ent_code: ent_code,
                        nickName: nickName,
                        icon: icon
                    };
                    forumUserService.createUser(entity, function(err, doc) {
                        if (err) {
                            console.error(err);
                            callback(err, null);
                        } else {
                            callback(null, doc);
                        }
                    });
                }
            },
            function(user, callback) {//获取板块信息
                forumThreadService.getByConditionsSelectyField({_id: new mongoose.Types.ObjectId(pid), ent_code: ent_code}, "info", function(err, doc) {
                    if (err || !doc) {
                        console.error(err);
                        callback(err, null, null);
                    } else {
                        callback(err, user, doc.info);
                    }
                });
            },
            function(user, infoId, callback) {//创建文章
                thread_entity.from = user._id;
                thread_entity.info = infoId
                forumThreadService.createThread(thread_entity, function(err, entity) {
                    if (err) {
                        console.error(err);
                        callback(err, null);
                    } else {
                        httpService.sendRequest(ent_code, mid, 'post');
                        forumAboutMEService.saveThread(entity);
                        callback(null, entity);
                    }
                });
            }
        ], function(err, thread) {
            if(err){
                res.json(returnCode.BUSY);
            }else{
                var rs = {
                    data:thread
                }
                res.json(_.assign(rs, returnCode.SUCCESS));
            }
        });
    }else{
        res.json(returnCode.WRONG_PARAM);
    }
});