'use strict';
var express = require('express'),
    router = express.Router(),
    returnCode = require('../../utils/returnCode'),
    then = require('thenjs'),
    util = require('../../utils/util'),
    _ = require('lodash');

var mongoose = require('mongoose');
var ForumInfo = mongoose.model('ForumInfo');
var forumInfoService = require('../../service/forumInfoService');
var forumThreadService = require('../../service/forumThreadService');
var forumUserThreadControlService = require('../../service/forumUserThreadControlService');
var forumTagService = require('../../service/forumTagService');
var forumCommentService = require('../../service/forumCommentService');
var forumThreadAttendService = require('../../service/forumThreadAttendService');
var httpService = require('../../service/httpService');
var redisPraiseLog = require('../../utils/redisPraiseLog');
var forumUserService = require('../../service/forumUserService');
var forumFollowInfoService = require('../../service/forumFollowInfoService');

var user = require('../../utils/user');

var async = require('async');

module.exports = function(app) {
    app.use('/v1/forum', router);
};

//论坛板块
router.get('/info/list/all', function(req, res, next) {
    var rs = {};
    var ent_code = req.session.user.ent_code;
    forumInfoService.getAllByGid({
        ent_code: ent_code
    }, 1, 100, function(err, datas) {
        if (err) {
            console.error(err);
            res.json(returnCode.BUSY);
        } else {
            rs.data = datas.items;
            res.json(_.assign(rs, returnCode.SUCCESS));
        }
    });
});

//新增论坛板块
router.post('/info/create', function(req, res, next) {
    var rs = {};
    req.body.ent_code = req.session.user.ent_code;
    forumInfoService.createInfo(req.body, function(err, info) {
        if (err) {
            console.error(err);
            res.json(returnCode.BUSY);
        } else {
            rs.data = {
                'id': info._id
            };
            res.json(_.assign(rs, returnCode.SUCCESS));
        }
    });
});


//获取目标论坛板块
router.get('/info/:fid/get', function(req, res, next) {
    var fid = req.params.fid || null;
    if (fid) {
        var ent_code = req.session.user.ent_code;
        var today = new Date();
        var source = req.session.mobileForumUser.source;
        httpService.createLog(req, source, fid, 2);
        forumInfoService.updateInfoPvCount(fid, function(err, result) {
            if (err) {
                console.error(err);
            } 
        });
        async.parallel([
            function(callback) {
                forumInfoService.getInfoById(fid, function(err, info) {
                    if (err) {
                        callback(err, null);
                    } else {
                        callback(null, info);
                    }
                });
            },
            function(callback) {
                var conditions = {
                    ent_code: ent_code,
                    info: fid,
                    level: 1,
                    status: 1,
                    $or: [{
                        isEvent: {
                            $ne:1
                        }
                    }, {
                        isEvent: 1,
                        'event.eventStartTime' : {$lte : today}
                    }]
                };

                forumThreadService.getAllCountByFid(conditions, function(err, threadCount) {
                    if (err) {
                        callback(err, null);
                    } else {
                        callback(null, threadCount);
                    }
                });
            },
            function(callback) {
                var conditions = {
                    ent_code: ent_code,
                    info: fid
                };
                forumFollowInfoService.count(conditions, function(err, count) {
                    if (err) {
                        callback(err, null);
                    } else {
                        callback(null, count);
                    }
                });
            }
        ], function(err, results) {
            if (err) {
                console.error(err);
                res.json(returnCode.BUSY);
            } else {
                var rs = {};
                rs.data = results[0].toObject();
                rs.data.threadCount = results[1];
                rs.data.favor_by = results[2];
                res.json(_.assign(rs, returnCode.SUCCESS));
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }

});

//更新目标论坛板块
router.post('/info/:fid/update', function(req, res, next) {
    var fid = req.params.fid;
    if (fid) {
        forumInfoService.updateInfoById(fid, req.body, function(err, result) {
            if (err) {
                res.json(returnCode.BUSY);
            } else {
                res.json(returnCode.SUCCESS);
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//删除目标论坛板块
router.post('/info/:fid/delete', function(req, res, next) {
    var fid = req.params.fid;
    if (fid) {
        forumInfoService.deleteInfoById(fid, function(err, result) {
            if (err) {
                res.json(returnCode.BUSY);
            } else {
                res.json(returnCode.SUCCESS);
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

/**
 * [description]
 * @param  {[type]}
 * @param  {[type]}
 * @param  {[type]}
 * @return {[type]}
 */
router.get('/info/:fid/threads', function(req, res, next) {
    var pageNo = req.query.pageNo || 1;
    var pageSize = req.query.pageSize || 10;
    var fid = req.params.fid;
    var sort = {
        top:-1,
        top_order_idx:-1,
        topTime:-1,
        tag_top:-1,
        tag_topTime:-1,
        quality:-1,
        created:-1
    };
    var ent_code = req.session.user.ent_code;
    var today = new Date();
    var conditions = {
        ent_code: ent_code,
        level: 1,
        status: 1,
        $or: [{
            isEvent: {
                $ne:1
            }
        }, {
            isEvent: 1,
            'event.eventStartTime' : {$lte : today}
        }]
    };
    if (fid) {
        conditions.info = fid;
    }
    if (fid) {
        async.waterfall([
            function(callback) {
                forumUserThreadControlService.getUserThreadControlById(user.getMobileUser(req), function(err, doc) {
                    if (err) {
                        callback(err, null);
                    } else {
                        if (doc) {
                            callback(null, doc);
                        } else {
                            callback(null, null);
                        }
                    }
                });
            }
        ], function(err, result) {
            if (err) {
                console.error(err);
                res.json(returnCode.BUSY);
            } else {
                if (result) {
                    conditions._id = {
                        $nin: result.thread
                    };
                }
                forumThreadService.getAllThreadByFid(conditions, pageNo, pageSize, sort, function(err, results) {
                    if (err) {
                        console.error(err);
                        res.json(returnCode.BUSY);
                    } else {
                        util.loadLevel(req.session.user.ent_code, results.items, function() {
                            redisPraiseLog.get(ent_code, user.getMobileUser(req), 'thread', function(error, docs) {
                                _.forEach(results.items, function(d, i) {
                                    if (results.items[i].toObject) {
                                        results.items[i] = results.items[i].toObject();
                                    }
                                    results.items[i].isPraise = false;
                                    for (var k = docs.length - 1; k >= 0; k--) {
                                        if (results.items[i]._id == docs[k]) {
                                            results.items[i].isPraise = true;
                                            break;
                                        }
                                    };

                                });
                                res.json(_.assign(results, returnCode.SUCCESS));
                            });
                        });
                    }
                });
            }
        });

    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//获取论坛最热文章列表
router.get('/info/:fid/hotThreads', function(req, res, next) {
    var fid = req.params.fid || null;
    var pageNo = req.query.pageNo || 1;
    var pageSize = req.query.pageSize || 10;
    var today = new Date();
    var conditions = {
        ent_code: req.session.user.ent_code,
        level: 1,
        info: fid,
        status: {
            $ne: 3
        },
        $or: [{
            isEvent: {
                $ne:1
            }
        }, {
            isEvent: 1,
            'event.eventStartTime' : {$lte : today}
        }]
    }

    if (fid) {

        async.waterfall([
            function(callback) {
                forumUserThreadControlService.getUserThreadControlById(user.getMobileUser(req), function(err, doc) {
                    if (err) {
                        callback(err, null);
                    } else {
                        if (doc) {
                            callback(null, doc);
                        } else {
                            callback(null, null);
                        }
                    }
                });
            }
        ], function(err, result) {
            if (err) {
                console.error(err);
                res.json(returnCode.BUSY);
            } else {
                if (result) {
                    conditions._id = {
                        $nin: result.thread
                    };
                }
                forumThreadService.getAllThreadByFid(conditions, pageNo, pageSize, '-comment_count -praise_count', function(err, results) {
                    if (err) {
                        console.error(err);
                        res.json(returnCode.BUSY);
                    } else {
                        res.json(_.assign(results, returnCode.SUCCESS));
                    }
                });
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//获取论坛晒图文章列表
router.get('/info/:fid/photoThreads', function(req, res, next) {
    var fid = req.params.fid || null;
    var pageNo = req.query.pageNo || 1;
    var pageSize = req.query.pageSize || 10;
    var today = new Date();
    var conditions = {
        ent_code: req.session.user.ent_code,
        level: 1,
        info: fid,
        type: 3,
        status: {
            $ne: 3
        },
        $or: [{
            isEvent: {
                $ne:1
            }
        }, {
            isEvent: 1,
            'event.eventStartTime' : {$lte : today}
        }]
    };

    if (fid) {

        async.waterfall([
            function(callback) {
                forumUserThreadControlService.getUserThreadControlById(user.getMobileUser(req), function(err, doc) {
                    if (err) {
                        callback(err, null);
                    } else {
                        if (doc) {
                            callback(null, doc);
                        } else {
                            callback(null, null);
                        }
                    }
                });
            }
        ], function(err, result) {
            if (err) {
                console.error(err);
                res.json(returnCode.BUSY);
            } else {
                if (result) {
                    conditions._id = {
                        $nin: result.thread
                    };
                }
                forumThreadService.getAllThreadByFid(conditions, pageNo, pageSize, null, function(err, results) {
                    if (err) {
                        console.error(err);
                        res.json(returnCode.BUSY);
                    } else {
                        res.json(_.assign(results, returnCode.SUCCESS));
                    }
                });
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//获取我的文章列表
router.get('/info/:fid/myThreads', function(req, res, next) {
    var fid = req.params.fid || null;
    var pageNo = req.query.pageNo || 1;
    var pageSize = req.query.pageSize || 10;

    var conditions = {
        ent_code: req.session.user.ent_code,
        from: user.getMobileUser(req),
        info: fid
    };

    if (fid) {

        async.waterfall([
            function(callback) {
                callback();
                // forumUserThreadControlService.getUserThreadControlById(user.getMobileUser(req), function(err, doc) {
                //     if (err) {
                //         callback(err, null);
                //     } else {
                //         if (doc) {
                //             callback(null, doc);
                //         } else {
                //             callback(null, null);
                //         }
                //     }
                // });
            }
        ], function(err, result) {
            if (err) {
                console.error(err);
                res.json(returnCode.BUSY);
            } else {
                if (result) {
                    conditions._id = {
                        $nin: result.thread
                    };
                }
                //获取子话题数据
                forumThreadService.getAllThreadByFid(conditions, pageNo, pageSize, '-quality -created', function(err, results) {
                    if (err) {
                        console.error(err);
                        res.json(returnCode.BUSY);
                    } else {

                        res.json(_.assign(results, returnCode.SUCCESS));
                    }
                });
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//获取我的文章列表
router.get('/info/myThreads', function(req, res, next) {
    var pageNo = req.query.pageNo || 1;
    var pageSize = req.query.pageSize || 10;

    var conditions = {
        ent_code: req.session.user.ent_code,
        from: user.getMobileUser(req),
        status: 1
    };

    async.waterfall([
        function(callback) {
            callback();
            // forumUserThreadControlService.getUserThreadControlById(user.getMobileUser(req), function(err, doc) {
            //     if (err) {
            //         callback(err, null);
            //     } else {
            //         if (doc) {
            //             callback(null, doc);
            //         } else {
            //             callback(null, null);
            //         }
            //     }
            // });
        }
    ], function(err, result) {
        if (err) {
            console.error(err);
            res.json(returnCode.BUSY);
        } else {
            if (result) {
                conditions._id = {
                    $nin: result.thread
                };
            }
            //获取子话题数据
            forumThreadService.getAllThreadByFid(conditions, pageNo, pageSize, '-quality -created', function(err, results) {
                if (err) {
                    console.error(err);
                    res.json(returnCode.BUSY);
                } else {
                    //判断是否已经点赞
                    util.loadLevel(req.session.user.ent_code, results.items, function() {
                        redisPraiseLog.get(req.session.user.ent_code, user.getMobileUser(req), 'thread', function(error, docs) {
                            _.forEach(results.items, function(d, i) {
                                if (results.items[i].toObject) {
                                    results.items[i] = results.items[i].toObject();
                                }
                                results.items[i].isPraise = false;
                                for (var k = docs.length - 1; k >= 0; k--) {
                                    if (results.items[i]._id == docs[k]) {
                                        results.items[i].isPraise = true;
                                        break;
                                    }
                                };
                            });
                            res.json(_.assign(results, returnCode.SUCCESS));
                        });
                    });

                }
            });
        }
    });
});

//获取他的文章列表
router.get('/info/hisThreads/:id', function(req, res, next) {
    var pageNo = req.query.pageNo || 1;
    var pageSize = req.query.pageSize || 10;

    var conditions = {
        ent_code: req.session.user.ent_code,
        from: req.params.id,
        status: 1
    };

    async.waterfall([
        function(callback) {
            callback();
            // forumUserThreadControlService.getUserThreadControlById(user.getMobileUser(req), function(err, doc) {
            //     if (err) {
            //         callback(err, null);
            //     } else {
            //         if (doc) {
            //             callback(null, doc);
            //         } else {
            //             callback(null, null);
            //         }
            //     }
            // });
        }
    ], function(err, result) {
        if (err) {
            console.error(err);
            res.json(returnCode.BUSY);
        } else {
            if (result) {
                conditions._id = {
                    $nin: result.thread
                };
            }
            //获取子话题数据
            forumThreadService.getAllThreadByFid(conditions, pageNo, pageSize, '-quality -created', function(err, results) {
                if (err) {
                    console.error(err);
                    res.json(returnCode.BUSY);
                } else {
                    //判断是否已经点赞
                    util.loadLevel(req.session.user.ent_code, results.items, function() {
                        redisPraiseLog.get(req.session.user.ent_code, user.getMobileUser(req), 'thread', function(error, docs) {
                            _.forEach(results.items, function(d, i) {
                                if (results.items[i].toObject) {
                                    results.items[i] = results.items[i].toObject();
                                }
                                results.items[i].isPraise = false;
                                for (var k = docs.length - 1; k >= 0; k--) {
                                    if (results.items[i]._id == docs[k]) {
                                        results.items[i].isPraise = true;
                                        break;
                                    }
                                };
                            });
                            res.json(_.assign(results, returnCode.SUCCESS));
                        });
                    });
                }
            });
        }
    });
});

//获取我的评论列表
router.get('/info/myComments', function(req, res, next) {
    var fid = req.params.fid || null;
    var pageNo = req.query.pageNo || 1;
    var pageSize = req.query.pageSize || 10;
    var id = user.getMobileUser(req);
    var conditions = {
        ent_code: req.session.user.ent_code,
        status: 1,
        //$or:[{from: id},{to: id}]
        to: id
    };
    forumCommentService.getMyComment(conditions, pageNo, pageSize, function(err, results) {
        if (err) {
            console.error(err);
            res.json(returnCode.BUSY);
        } else {
            res.json(_.assign(results, returnCode.SUCCESS));
        }
    });
});


//获取跟我相关的评论列表
router.get('/info/:tid/comment/list', function(req, res, next) {
    var tid = req.params.tid || null;
    var pageNo = req.query.pageNo || 1;
    var pageSize = req.query.pageSize || 10;
    var id = user.getMobileUser(req);
    var conditions = {
        ent_code: req.session.user.ent_code,
        thread: tid, //5539fa026cf87854c87be3c9
        level: '1'
    };
    if (!tid) {
        return res.json(returnCode.WRONG_PARAM);
    }
    forumCommentService.getAllCommentByThread(conditions, function(err, results) {
        if (err) {
            console.error(err);
            res.json(returnCode.BUSY);
        } else {
            var docs = [];
            if (results && results.length > 0) {
                results.forEach(function(doc, i) {
                    //过滤与自己无关的评论
                    if (doc.from._id != id && doc.thread.from != id) { //文章和一级评论的作者不是自己
                        if (doc.comments && doc.comments.length > 0) { //有子评论
                            var comments = [];
                            doc.comments.forEach(function(comment, j) {
                                if ((comment.from && comment.from._id == id) || (comment.to && comment.to._id == id)) {
                                    //二级评论的作者 或 被评论者是自己
                                    comments.push(comment);
                                }
                            });
                            if (comments.length > 0) {
                                doc.comments = comments;
                                docs.push(doc);
                            }
                        }
                    } else { //文章或一级评论的作者是自己那么所有的子评论都不用过滤
                        docs.push(doc);
                    }
                });
            }
            var obj = {};
            obj.total = docs.length;
            obj.pageNo = pageNo;
            obj.pageSize = pageSize;
            obj.items = docs.slice((pageNo - 1 >= 0 ? pageNo - 1 : 0) * pageSize, pageSize);
            res.json(_.assign(obj, returnCode.SUCCESS));
        }
    });
});

//搜索文章列表
router.get('/info/:fid/serachThreads', function(req, res, next) {
    var fid = req.params.fid || null;
    var pageNo = req.query.pageNo || 1;
    var pageSize = req.query.pageSize || 10;
    var content = req.query.content;
    var ent_code = req.session.user.ent_code;
    var today = new Date();
    var conditions = {
        ent_code: ent_code,
        status: 1,
        info: fid,
        $or: [{
            isEvent: {
                $ne:1
            }
        }, {
            isEvent: 1,
            'event.eventStartTime' : {$lte : today}
        }]
    };

    if (content) {
        conditions.content = {
            $regex: content,
            $options: 'i'
        };
    }

    if (fid) {

        async.waterfall([
            function(callback) {
                forumUserThreadControlService.getUserThreadControlById(user.getMobileUser(req), function(err, doc) {
                    if (err) {
                        callback(err, null);
                    } else {
                        if (doc) {
                            callback(null, doc);
                        } else {
                            callback(null, null);
                        }
                    }
                });
            }
        ], function(err, result) {
            if (err) {
                console.error(err);
                res.json(returnCode.BUSY);
            } else {
                if (result) {
                    conditions._id = {
                        $nin: result.thread
                    };
                }
                //获取子话题数据
                forumThreadService.getAllThreadByFid(conditions, pageNo, pageSize, '-praise_count', function(err, results) {
                    if (err) {
                        console.error(err);
                        res.json(returnCode.BUSY);
                    } else {
                        //判断是否已经点赞
                        util.loadLevel(req.session.user.ent_code, results.items, function() {
                            redisPraiseLog.get(ent_code, user.getMobileUser(req), 'thread', function(error, docs) {
                                _.forEach(results.items, function(d, i) {
                                    if (results.items[i].toObject) {
                                        results.items[i] = results.items[i].toObject();
                                    }
                                    results.items[i].isPraise = false;
                                    for (var k = docs.length - 1; k >= 0; k--) {
                                        if (results.items[i]._id == docs[k]) {
                                            results.items[i].isPraise = true;
                                            break;
                                        }
                                    };
                                });
                                res.json(_.assign(results, returnCode.SUCCESS));
                            });
                        });

                    }
                });
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//获取标签列表
router.get('/info/:fid/tags', function(req, res, next) {
    var fid = req.params.fid || null;
    var pageNo = req.query.pageNo || 1;
    var pageSize = req.query.pageSize || 1000;

    var conditions = {
        ent_code: req.session.user.ent_code,
        info: fid
    };
    if (fid) {
        forumTagService.getAllTag(conditions, pageNo, pageSize, function(err, results) {
            if (err) {
                console.error(err);
                res.json(returnCode.BUSY);
            } else {
                res.json(_.assign(results, returnCode.SUCCESS));
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//查询前台可使用的所有标签（针对发帖）
router.get('/info/:fid/mobileList', function(req, res, next) {
    var fid = req.params.fid || null;
    var pageNo = req.query.pageNo || 1;
    var pageSize = req.query.pageSize || 10;
    var conditions = {
        ent_code: req.session.user.ent_code,
        type: 1
    };
    if (fid) {
        forumTagService.getAllTag(conditions, pageNo, pageSize, function(err, results) {
            if (err) {
                console.error(err);
                res.json(returnCode.BUSY);
            } else {
                res.json(_.assign(results, returnCode.SUCCESS));
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//我的评论
router.get('/info/myAttend', function(req, res, next) {
    var user_id = user.getMobileUser(req);
    var pageNo = req.query.pageNo,
        pageSize = req.query.pageSize;
    var total = 0;
    then(function(cont) {
        var conditions = {
            attend: user_id
        }
        forumThreadAttendService.findMyAttentByPage(conditions, pageNo, pageSize, '-updated', function(err, attends) {
            if (err) {
                cont(err);
            } else {
                if (attends && attends.datas.length > 0) {
                    total = attends.total;
                    var threadIds = [];
                    _.forEach(attends.datas, function(a) {
                        threadIds.push(a.thread);
                    });
                    cont(null, threadIds);
                } else {
                    res.json(_.assign({
                        data: [],
                        total: 0
                    }, returnCode.SUCCESS));
                }
            }
        });
    }).then(function(cont, threadIds) {
        forumUserThreadControlService.getUserThreadControlById(user.getMobileUser(req), function(err, doc) {
            cont(err, doc, threadIds);
        });
    }).then(function(cont, results, threadIds) {
        var conditions = {};
        if (results) {
            conditions._id = {
                $in: threadIds
                    // $nin:results
            }
        } else {
            conditions._id = {
                $in: threadIds
            }
        }
        forumThreadService.getAllThreadByFid(conditions, pageNo, pageSize, '', function(err, threads) {
            cont(err, threads);
        });
    }).then(function(cont, threads) {
        threads.total = total;
        res.json(_.assign(threads, returnCode.SUCCESS));
    }).then(function(cont) {

    }).fail(function(cont, err) {
        console.error(err);
        res.json(returnCode.BUSY);
    })
});


//论坛板块
router.get('/info/list/allWithThreadCount', function(req, res, next) {
    var rs = {};
    var ent_code = req.session.user.ent_code;
    var today = new Date();
    async.waterfall([
        function(callback) {
            forumInfoService.getAllByGid({ent_code: ent_code}, 1, 100, function(err, datas) {
                callback(err, datas.items);
            });
        },
        function(infos, callback) { //获取关注数
            var asyncTasks = [];
            _.forEach(infos, function(info, i) {
                asyncTasks.push(function(cont) {
                    var conditions = {
                        ent_code: ent_code,
                        info: info._id
                    };
                    forumFollowInfoService.count(conditions, function(err, count) {
                        if (err) {
                            console.error(err);
                            cont(err, null);
                        } else {
                            if (infos[i].toObject) {
                                infos[i] = infos[i].toObject();
                            }
                            infos[i].favor_by = count;
                            cont(null, info);
                        }
                    });
                });
            });
            then.parallel(asyncTasks).then(function(cont, datas) {
                callback(null, infos);
            }).fail(function(err, cont) {
                callback(err, null);
            });
        },
        function(infos, callback) { //获取用户
            var asyncTasks = [];
            _.forEach(infos, function(info, i) {
                asyncTasks.push(function(cont) {
                    var conditions = {
                        ent_code: ent_code,
                        info: info._id,
                        level: 1,
                        status: 1,
                        $or: [{
                            isEvent: {
                                $ne:1
                            }
                        }, {
                            isEvent: 1,
                            'event.eventStartTime' : {$lte : today}
                        }]
                    };
                    forumThreadService.getAllCountByFid(conditions, function(err, count) {
                        if (err) {
                            console.error(err);
                            cont(err, null);
                        } else {
                            if (infos[i].toObject) {
                                infos[i] = infos[i].toObject();
                            }
                            infos[i].threadCount = count;
                            cont(null, info);
                        }
                    });
                });
            });
            then.parallel(asyncTasks).then(function(cont, datas) {
                callback(null, infos);
            }).fail(function(err, cont) {
                callback(err, null);
            });
        }
    ], function(err, result) {
        if (err) {
            console.error(err);
            res.json(returnCode.BUSY);
        } else {
            rs.data = result;
            res.json(_.assign(rs, returnCode.SUCCESS));
        }
    });
});

//搜索文章列表，不带板块ID
router.get('/info/serachThreads', function(req, res, next) {
    var pageNo = req.query.pageNo || 1;
    var pageSize = req.query.pageSize || 10;
    var content = req.query.content;
    var ent_code = req.session.user.ent_code;
    var today = new Date();
    var conditions = {
        ent_code: ent_code,
        status: 1,
        $or: [{
            isEvent: {
                $ne:1
            }
        }, {
            isEvent: 1,
            'event.eventStartTime' : {$lte : today}
        }]
    };

    if (content) {
        conditions.content = {
            $regex: content,
            $options: 'i'
        };
    }

    async.waterfall([
        function(callback) {
            forumUserThreadControlService.getUserThreadControlById(user.getMobileUser(req), function(err, doc) {
                if (err) {
                    callback(err, null);
                } else {
                    if (doc) {
                        callback(null, doc);
                    } else {
                        callback(null, null);
                    }
                }
            });
        }
    ], function(err, result) {
        if (err) {
            console.error(err);
            res.json(returnCode.BUSY);
        } else {
            if (result) {
                conditions._id = {
                    $nin: result.thread
                };
            }
            //获取子话题数据
            forumThreadService.getAllThreadByFid(conditions, pageNo, pageSize, '-praise_count', function(err, results) {
                if (err) {
                    console.error(err);
                    res.json(returnCode.BUSY);
                } else {
                    //判断是否已经点赞
                    util.loadLevel(req.session.user.ent_code, results.items, function() {
                        redisPraiseLog.get(ent_code, user.getMobileUser(req), 'thread', function(error, docs) {
                            _.forEach(results.items, function(d, i) {
                                if (results.items[i].toObject) {
                                    results.items[i] = results.items[i].toObject();
                                }
                                results.items[i].isPraise = false;
                                for (var k = docs.length - 1; k >= 0; k--) {
                                    if (results.items[i]._id == docs[k]) {
                                        results.items[i].isPraise = true;
                                        break;
                                    }
                                };
                            });
                            res.json(_.assign(results, returnCode.SUCCESS));
                        });
                    });

                }
            });
        }
    });
});

//获取用户收藏的帖子,活动贴
router.get('/info/getUserFavorInfo', function(req, res, next) {
    var pageNo = req.query.pageNo || 1;
    var pageSize = req.query.pageSize || 10;
    var userId = user.getMobileUser(req);
    var ent_code = req.session.user.ent_code;
    var q = {
        ent_code: ent_code,
        from: userId
    };

    forumFollowInfoService.findAll(q, null, null, function(err, doc){
        if(err || !doc){
            console.error(err);
            res.json(returnCode.BUSY);
        }else {
            var info_ids = [];
            for(var i = doc.length - 1; i >= 0; i--){
                info_ids.push(doc[i].info);
            }
            var rs = {};
            var today = new Date();
            async.waterfall([
                function(callback) {
                    var info_condition = {
                        ent_code: ent_code,
                        _id:{$in:info_ids}
                    }
                    forumInfoService.getAllByGid(info_condition, pageNo, pageSize, function(err, datas) {
                        callback(err, datas.items);
                    });
                },
                function(infos, callback) { //获取关注数
                    var asyncTasks = [];
                    _.forEach(infos, function(info, i) {
                        asyncTasks.push(function(cont) {
                            var conditions = {
                                ent_code: ent_code,
                                info: info._id
                            };
                            forumFollowInfoService.count(conditions, function(err, count) {
                                if (err) {
                                    console.error(err);
                                    cont(err, null);
                                } else {
                                    if (infos[i].toObject) {
                                        infos[i] = infos[i].toObject();
                                    }
                                    infos[i].favor_by = count;
                                    cont(null, info);
                                }
                            });
                        });
                    });
                    then.parallel(asyncTasks).then(function(cont, datas) {
                        callback(null, infos);
                    }).fail(function(err, cont) {
                        callback(err, null);
                    });
                },
                function(infos, callback) { //获取用户
                    var asyncTasks = [];
                    _.forEach(infos, function(info, i) {
                        asyncTasks.push(function(cont) {
                            var conditions = {
                                ent_code: ent_code,
                                info: info._id,
                                level: 1,
                                status: 1,
                                $or: [{
                                    isEvent: {
                                        $ne:1
                                    }
                                }, {
                                    isEvent: 1,
                                    'event.eventStartTime' : {$lte : today}
                                }]
                            };
                            forumThreadService.getAllCountByFid(conditions, function(err, count) {
                                if (err) {
                                    console.error(err);
                                    cont(err, null);
                                } else {
                                    if (infos[i].toObject) {
                                        infos[i] = infos[i].toObject();
                                    }
                                    infos[i].threadCount = count;
                                    cont(null, info);
                                }
                            });
                        });
                    });
                    then.parallel(asyncTasks).then(function(cont, datas) {
                        callback(null, infos);
                    }).fail(function(err, cont) {
                        callback(err, null);
                    });
                }
            ], function(err, result) {
                if (err) {
                    console.error(err);
                    res.json(returnCode.BUSY);
                } else {
                    rs.data = result;
                    res.json(_.assign(rs, returnCode.SUCCESS));
                }
            });

        }
    });
});


//获取用户收藏的板块的ID return  Map
router.get('/info/getUserFavorInfoIds', function(req, res, next) {
    var userId = user.getMobileUser(req);
    var ent_code = req.session.user.ent_code;
    var q = {
        ent_code: ent_code,
        from: userId
    };
    forumFollowInfoService.findAll(q, null, 'info', function(err, results){
        if(err || !results){
            console.error(err);
            res.json(returnCode.BUSY);
        }else {
            var map = { //返回map对象，方便前端查找
                data:{}
            };
            _.forEach(results, function(d){
                map.data[d.info] = 1;
            });
            res.json(_.assign(map, returnCode.SUCCESS));
        }
    });
});