'use strict';
var express = require('express'),
    router = express.Router(),
    returnCode = require('../../utils/returnCode'),
    _ = require('lodash');

var mongoose = require('mongoose');
var forumUserService = require('../../service/forumUserService');
var forumThreadService = require('../../service/forumThreadService');

var forumFollowThreadService = require('../../service/forumFollowThreadService');
var forumFollowInfoService = require('../../service/forumFollowInfoService');
var forumFollowUserService = require('../../service/forumFollowUserService');

var async = require('async');
var util = require('../../utils/util');
var user = require('../../utils/user');

module.exports = function(app) {
    app.use('/v1/forum', router);
};

//更新用户收藏信息
var updateuser = function(condition, entity, callback){
	forumUserService.updateByIdWithOptions(condition, entity, {upsert: false}, function(err, user_id) {
        if (err) {
            callback(err, null);
        } else {
            callback(null, user_id);
        }
    });
};

//查询用户是否已收藏(板块/文章/用户)
var queryIsFavor = function(userID, field, subField, condition, callback){
	forumUserService.getUserById(userID, function(err, doc) {
        if (err) {
            callback(err, null);
        } else {
        	if(doc){
        		var list = doc[field];
                var flag = null;
                _.forEach(list, function(one, i) {
                    if(one[subField] == condition){
                        flag = 1;
                    }
                });
                callback(flag); 
        	}else{
        		callback('查询用户信息错误');
        	}
        }
    });
};

//获取用户信息
router.get('/user/:fid/get', function(req, res, next) {
    var fid = req.params.fid || null;
    var rs = {};
    async.parallel([
        function(cb) {
            forumUserService.getUserById(user.getMobileUser(req), function(err, entity) {
                if (err) {
                    cb(err, null);
                } else {
                    cb(null, entity);
                }
            });
        },
        function(cb) {
            var conditions = {
                ent_code: req.session.user.ent_code,
                from: user.getMobileUser(req),
                status: 1,
                info: fid
            };
            forumThreadService.getAllCountByFid(conditions, function(err, count) {
                if (err) {
                    cb(err, null);
                } else {
                    cb(null, count);
                }
            });
        }
    ], function(err, results) {
        if (err) {
            console.error(err);
            res.json(returnCode.BUSY);
        } else {
            rs.data = results[0] || {};
            rs.myThreadCount = results[1] || 0;
            res.json(_.assign(rs, returnCode.SUCCESS));
        }
    });
});

//更新用户信息
router.post('/user/:uid/update', function(req, res, next) {
    var uid = req.params.uid;
    if (uid) {
        forumUserService.updateUserById(uid, req.body, function(err, result) {
            if (err) {
                res.json(returnCode.BUSY);
            } else {
                res.json(returnCode.SUCCESS);
            }
        });
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//收藏板块
router.post('/user/favorInfo/:infoID', function(req, res, next) {
    var userID = user.getMobileUser(req);
    var ent_code = req.session.user.ent_code;
    var infoID = req.params.infoID;
    if (userID && infoID) {
        async.waterfall([
        	function(callback) {
		        forumFollowInfoService.isNotFollow({from: userID, info:infoID, ent_code: ent_code}, callback);
		    },
		    function(callback) {
		    	var obj = {
                    from: userID, 
                    info: infoID, 
                    ent_code: ent_code
                };
		    	forumFollowInfoService.create(obj, function(err, doc){
                    callback(err, null);
                });
		    }
		], function (err, result) {
		    if (err) {
                if(err == 1){
		    		var rs = {
					  'errorcode': 1,
					  'errormsg': '已收藏'
					}
		    		res.json(rs);
		    	}else{
		    		res.json(returnCode.BUSY);
		    	}
            } else {
                var rs = {
                    data:true
                }
                res.json(_.assign(rs, returnCode.SUCCESS));
            }
		});
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//取消收藏板块
router.post('/user/disfavorInfo/:infoID', function(req, res, next) {
    var userID = user.getMobileUser(req);
    var ent_code = req.session.user.ent_code;
    var infoID = req.params.infoID;
    if (userID && infoID) {
        async.waterfall([
		    function(callback) {
		    	forumFollowInfoService.remove({from: userID, info:infoID, ent_code: ent_code}, function(err, result){
                    callback(err, result);
                });
		    }
		], function (err, result) {
		    if (err) {
                res.json(returnCode.BUSY);
            } else {
                var rs = {
                    data:false
                }
                res.json(_.assign(rs, returnCode.SUCCESS));
            }
		});
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//收藏文章
router.post('/user/favorThread/:threadID', function(req, res, next) {
    var userID = user.getMobileUser(req);
    var ent_code = req.session.user.ent_code;
    var threadID = req.params.threadID;
    if (userID && threadID) {
        async.waterfall([
        	function(callback) {
                forumFollowThreadService.isNotFollow({from: userID, thread: threadID, ent_code: ent_code}, callback);
            },
            function(callback) {
                var obj = {
                    from: userID, 
                    thread: threadID, 
                    ent_code: ent_code
                };
                forumFollowThreadService.create(obj, function(err, doc){
                    callback(err, null);
                });
            }
		], function (err, result) {
		    if (err) {
                if(err == 1){
		    		var rs = {
					  'errorcode': 1,
					  'errormsg': '已收藏'
					}
		    		res.json(rs);
		    	}else{
                    res.json(returnCode.BUSY);
		    	}
            } else {
                var rs = {
                    data:true
                }
                res.json(_.assign(rs, returnCode.SUCCESS));
            }
		});
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//取消收藏文章
router.post('/user/disfavorThread/:threadID', function(req, res, next) {
    var userID = user.getMobileUser(req);
    var ent_code = req.session.user.ent_code;
    var threadID = req.params.threadID;
    if (userID && threadID) {
        async.waterfall([
		    function(callback) {
		    	forumFollowThreadService.remove({from: userID, thread: threadID, ent_code: ent_code}, function(err, result){
                    callback(err, result);
                });
		    }
		], function (err, result) {
		    if (err) {
                res.json(returnCode.BUSY);
            } else {
                var rs = {
                    data:false
                }
                res.json(_.assign(rs, returnCode.SUCCESS));
            }
		});
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//收藏用户
router.post('/user/favorUser/:favorUserID', function(req, res, next) {
    var userID = user.getMobileUser(req);// 收藏的用户
    var ent_code = req.session.user.ent_code;
    var favorUserID = req.params.favorUserID; //被收藏的用户
    if (userID && favorUserID) {
        async.waterfall([
        	function(callback) {
                forumFollowUserService.isNotFollow({from: userID, to: favorUserID, ent_code: ent_code}, callback);
            },
            function(callback) {
                var obj = {
                    from: userID, 
                    to: favorUserID, 
                    ent_code: ent_code
                };
                forumFollowUserService.create(obj, function(err, doc){
                    callback(err, null);
                });
            }
		], function (err, result) {
		    if (err) {
		    	if(err == 1){
		    		var rs = {
					  'errorcode': 1,
					  'errormsg': '已收藏'
					}
		    		res.json(rs);
		    	}else{
		    		res.json(returnCode.BUSY);
		    	}
            } else {
                var rs = {
                    data:true
                }
                res.json(_.assign(rs, returnCode.SUCCESS));
            }
		});
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//取消收藏用户
router.post('/user/disfavorUser/:favorUserID', function(req, res, next) {
    var userID = user.getMobileUser(req);// 收藏的用户
    var ent_code = req.session.user.ent_code;
    var favorUserID = req.params.favorUserID; //被收藏的用户
    if (userID && favorUserID) {
        async.waterfall([
		    function(callback) {
		    	forumFollowUserService.remove({from: userID, to: favorUserID, ent_code: ent_code}, function(err, result){
                    callback(err, result);
                });
		    }
		], function (err, result) {
		    if (err) {
                res.json(returnCode.BUSY);
            } else {
                var rs = {
                    data:false
                }
                res.json(_.assign(rs, returnCode.SUCCESS));
            }
		});
    } else {
        res.json(returnCode.WRONG_PARAM);
    }
});

//获取关注的用户
router.get('/user/getUserFavorUser', function(req, res, next) {
    var pageNo = req.query.pageNo || 1;
    var pageSize = req.query.pageSize || 10;
    var userId = user.getMobileUser(req);
    var ent_code = req.session.user.ent_code;
    var q = {
        ent_code: ent_code,
        from: userId
    };
    forumFollowUserService.find(pageNo, pageSize, q, null, function(err, results){
        if(err){
            console.error(err);
            res.json(returnCode.BUSY);
        }else{
            util.loadLevel(ent_code, results.items, function() {
                var asyncTasks = [];
                _.forEach(results.items, function(d, i) {
                    asyncTasks.push(function(cont) {
                        var conditions = {
                            ent_code: ent_code,
                            from: d.to._id,
                            status: 1
                        };
                        forumThreadService.getByConditionsSelectyFieldWithOpentions(conditions, 'title content info', 1, 0, '-_id',  function(err, thread) {
                            if (err) {
                                console.error(err);
                                cont(err, null);
                            } else {
                                if(d.toObject){
                                    d = d.toObject();
                                }
                                d.thread = thread;
                                cont(null, d);
                            }
                        });
                    });
                });
                async.parallel(asyncTasks, function(err, datas){
                    if(err){
                        console.error(err);
                        res.json( returnCode.BUSY);
                    }else{
                        results.items = datas;
                        res.json(_.assign(results, returnCode.SUCCESS));
                    }
                });
            }); 
        }
    });
});

//查看用户信息
router.get('/user/getUserInfo/:userId', function(req, res, next) {
    var userId = req.params.userId;
    var my_userID = user.getMobileUser(req);
    var ent_code = req.session.user.ent_code;

    async.waterfall([function(callback) {
        //获取用户荣誉头衔
        var user_conditions = {
            ent_code: ent_code,
            _id: userId
        };
        forumUserService.getByConditionsSelectyField(user_conditions, 'uid mid nickName icon honorTitles', function(err, doc) {
            callback(err, doc);
        });
    }, function(forumUser, callback) {
        async.parallel([
            function(callback) {
                //获取用户关注用户数
                var follow_conditions = {
                    ent_code: ent_code,
                    from: userId
                };
                forumFollowUserService.count(follow_conditions, callback);
            },
            function(callback) {
                //获取用户被关注数
                var follow_by_conditions = {
                    ent_code: ent_code,
                    to: userId
                };
                forumFollowUserService.count(follow_by_conditions, callback);
            },
            function(callback) {
                user.findMember(ent_code, forumUser.mid, callback);
            },
            function(callback) {
                //是否已关注当前用户
                forumFollowUserService.isNotFollow({from: my_userID, to: userId, ent_code: ent_code}, function(result){
                    if(result){
                        callback(null, true);
                    }else{
                        callback(null, false);
                    }
                });
            }
        ], function(err, results) {
            if (err) {
                console.error(err);
                callback(err, null);
            } else {
                if (forumUser.toObject) {
                    forumUser = forumUser.toObject();
                }
                if(results[2] && results[2][0]){
                    forumUser.integral = results[2][0].integral;
                    forumUser.exp = results[2][0].exp;
                    forumUser.rankName = results[2][0].rankName;
                }

                var object = {
                    forumUser : forumUser,
                    isFollow : results[3]
                }

                callback(null, object)
            }
        });
    }], function(err, result) {
        if(err){
            console.error(err);
            res.json(returnCode.BUSY);
        }else{
            var rs = {
                data: result
            };
            res.json(_.assign(rs, returnCode.SUCCESS));
        }
    });
});


//查看用户的收藏信息
router.get('/user/getUserFavorWithThreadAndInfo', function(req, res, next) {
    var userId = user.getMobileUser(req);
    var ent_code = req.session.user.ent_code;

    var q = {
        ent_code: ent_code,
        from: userId
    };

    async.parallel([
        function(callback) {
            //获取用户收藏的帖子
            forumFollowThreadService.findWithThread(q, 'isEvent event', function(err, docs){
                if(err || !docs){
                    console.error(err);
                    callback(err, null);
                }else{
                    var today = new Date();
                    var thread_count = 0;
                    var thread_event_count = 0;
                    _.forEach(docs, function(d, i){
                        if(d.thread){
                            if(d.thread.isEvent != 1){
                                thread_count++;
                            }else if(d.thread.isEvent == 1 && d.thread.event.eventStartTime <= today){
                                thread_event_count++;
                            }
                        }
                    });
                    var obj = {
                        follow_thread_count: thread_count,
                        follow_thread_event_count: thread_event_count
                    }
                    callback(null, obj);
                }
            });
        },
        function(callback) {
            //获取用户收藏的圈子
            forumFollowInfoService.count(q, callback);
        }
    ], function(err, results) {
        if(err){
            console.error(err);
            res.json(returnCode.BUSY);
        }else{
            results[0].follow_info_count = results[1];
            var rs = {
                data: results[0]
            };
            res.json(_.assign(rs, returnCode.SUCCESS));
        }
    });
});



//查看总经验排行
router.get('/user/exp/topChart', function(req, res, next) {
    var userId = user.getMobileUser(req);
    var ent_code = req.session.user.ent_code;
    forumUserService.getByConditionsSelectyField({
        _id:userId
    }, 'mid', function(err, doc){
        if(err){
            console.error(err);
            res.json(returnCode.BUSY);
        }else if(!doc.mid){
            console.error(err);
            res.json(returnCode.DATA_NOTEXITS);
        }else{
            async.parallel([
                function(cb) {
                    user.getMemberExp(ent_code, doc.mid, 3, function(err, entity) {
                        cb(err, entity);
                    });
                },
                function(cb) {
                    user.getMemberExpTopChart(ent_code, function(err, entity) {
                        cb(err, entity);
                    });
                }
            ], function(err, results) {
                if (err) {
                    console.error(err);
                    res.json(returnCode.BUSY);
                } else {
                    var rs = {
                        user:results[0],
                        list:results[1]
                    };
                    res.json(_.assign(rs, returnCode.SUCCESS));
                }
            });
        }
    });
});


//查看月经验排行
router.get('/user/exp/topMonthChart', function(req, res, next) {
    var userId = user.getMobileUser(req);
    var ent_code = req.session.user.ent_code;
    forumUserService.getByConditionsSelectyField({
        _id:userId
    }, 'mid', function(err, doc){
        if(err){
            console.error(err);
            res.json(returnCode.BUSY);
        }else if(!doc.mid){
            console.error(err);
            res.json(returnCode.DATA_NOTEXITS);
        }else{
            async.parallel([
                function(cb) {
                    user.getMemberExp(ent_code, doc.mid, 1, function(err, entity) {
                        cb(err, entity);
                    });
                },
                function(cb) {
                    user.getMemberExpTopMonthChart(ent_code, function(err, entity) {
                        cb(err, entity);
                    });
                }
            ], function(err, results) {
                if (err) {
                    console.error(err);
                    res.json(returnCode.BUSY);
                } else {
                    var rs = {
                        user:results[0],
                        list:results[1]
                    };
                    res.json(_.assign(rs, returnCode.SUCCESS));
                }
            });
        }
    });
});


//查看周经验排行
router.get('/user/exp/topWeekChart', function(req, res, next) {
    var userId = user.getMobileUser(req);
    var ent_code = req.session.user.ent_code;
    forumUserService.getByConditionsSelectyField({
        _id:userId
    }, 'mid', function(err, doc){
        if(err){
            console.error(err);
            res.json(returnCode.BUSY);
        }else if(!doc.mid){
            console.error(err);
            res.json(returnCode.DATA_NOTEXITS);
        }else{
            async.parallel([
                function(cb) {
                    user.getMemberExp(ent_code, doc.mid, 2,function(err, entity) {
                        cb(err, entity);
                    });
                },
                function(cb) {
                    user.getMemberExpTopWeekChart(ent_code, function(err, entity) {
                        cb(err, entity);
                    });
                }
            ], function(err, results) {
                if (err) {
                    console.error(err);
                    res.json(returnCode.BUSY);
                } else {
                    var rs = {
                        user:results[0],
                        list:results[1]
                    };
                    res.json(_.assign(rs, returnCode.SUCCESS));
                }
            });
        }
    });
});